from rest_framework import status
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status, permissions
from rest_framework_simplejwt.tokens import RefreshToken, AccessToken, UntypedToken
from rest_framework_simplejwt.exceptions import TokenError
from .serializers import UserUpdateSerializer, SubAuthSerializer, UserProfileSerializer
from .permissions import APIKeyPermission
import requests
from django.conf import settings
from .models import User
from datetime import timedelta
import base64
import time
from .utils import get_access_token,create_first_version, translate_file,get_buckets, upload_large_file_to_storage,create_storage_location,create_bucket, upload_file , translate_to_viewable , translate_to_svf
import tempfile , os , json
from django.http import JsonResponse, StreamingHttpResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods
from django.core.files.uploadedfile import UploadedFile

# Create your views here.


class UserProfileView(APIView):
    """
    Each user Retrieve his profile.
    """
    permission_classes = [permissions.IsAuthenticated]  
    serializer_class = UserProfileSerializer
    
    def get(self, request):
        user = request.user
        serializer = self.serializer_class(user)
        return Response(serializer.data, status=status.HTTP_200_OK)


class UserUpdateAPIView(APIView):
    permission_classes = [permissions.AllowAny , APIKeyPermission]

    def patch(self, request):
        user = request.user
        serializer = UserUpdateSerializer(user, data=request.data, partial=True)
        
        if serializer.is_valid():
            serializer.save()
            data = serializer.data
            data["is_active"] = user.is_active  
            return Response(data, status=status.HTTP_200_OK)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class UAEPassLoginURLView(APIView):
    permission_classes = [permissions.AllowAny , APIKeyPermission]
    def get(self, request):
        token = AccessToken()
        token['timestamp'] = int(time.time())
        token['purpose'] = 'uaepass_login'
        token.set_exp(lifetime=timedelta(minutes=5))  
        
        if request.GET.get('frontend_url'):
            settings.UAE_PASS['REDIRECT_URI'] = request.GET.get('frontend_url')
        
        auth_url = (
            settings.UAE_PASS['AUTHORIZATION_URL']+"?"
            f"response_type=code"
            f"&client_id={settings.UAE_PASS['CLIENT_ID']}"
            f"&scope=urn:uae:digitalid:profile:general"
            f"&state={str(token)}"
            f"&redirect_uri={settings.UAE_PASS['REDIRECT_URI']}"
            f"&acr_values={settings.UAE_PASS['ACR_VALUES']}"
        )

        return Response({
            'status' : True,
            'message': 'Login URL generated successfully.',
            'auth_url': auth_url,
        }) 


class UAEPassCallbackView(APIView):
    permission_classes = [permissions.AllowAny]
    def get(self, request):
        code = request.GET.get('code')
        state = request.GET.get('state')

        if not code or not state:
            return Response({'error': 'Missing code or state'}, status=status.HTTP_400_BAD_REQUEST)

        # Validate the SimpleJWT token from state
        try:
            UntypedToken(state)
        except TokenError:
            return Response({'error': 'Invalid or expired state token'}, status=status.HTTP_400_BAD_REQUEST)

        # Continue with your UAE Pass token exchange or next logic
      
        # Exchange code for access token
        token_response = self.exchange_code_for_token(code)
        if 'access_token' not in token_response:
            return Response({'error': 'Failed to obtain access token.'}, status=status.HTTP_400_BAD_REQUEST)

        access_token = token_response['access_token']

        # Fetch user info
        user_info = self.get_user_info(access_token)
        if 'sub' not in user_info:
            return Response({'error': 'Failed to fetch user info.'}, status=status.HTTP_400_BAD_REQUEST)

        # Save or update user
        user, created = User.objects.update_or_create(
            sub=user_info['sub'],
            defaults={
                'username': user_info.get('name', ''),
                'email': user_info.get('email'),
                'phone_number': user_info.get('mobile'),
                'emirates_id': user_info.get('idn'),
            }
        )

        user_type_display = user.get_user_type_display()
        
        if created:
            user.set_unusable_password()
            user.save()
            
        return Response({
            'status': True,
            'message': 'User authenticated successfully.',
            "token_data":self.generate_access_token(user),
            'user': {
                **user_info,
                'userType': {
                    'userTypeId': user.user_type,
                    'userTypeName': user_type_display
                }
            },
        })

    def exchange_code_for_token(self, code):
        client_id = settings.UAE_PASS['CLIENT_ID']
        client_secret = settings.UAE_PASS['CLIENT_SECRET']
        redirect_uri = settings.UAE_PASS['REDIRECT_URI']
        token_url = settings.UAE_PASS['TOKEN_URL']

        auth_str = f"{client_id}:{client_secret}"
        b64_auth_str = base64.b64encode(auth_str.encode()).decode()

        headers = {
            'Authorization': f'Basic {b64_auth_str}',
            'Content-Type': 'application/x-www-form-urlencoded',
        }
        data = {
            'grant_type': 'authorization_code',
            'code': code,
            'redirect_uri': redirect_uri,
        }

        try:
            print("Sending token request to:", token_url)
            print("Request headers:", headers)
            print("Request data:", data)
            
            response = requests.post(token_url, headers=headers, data=data,proxies={})
            print("Response status code:", response.status_code)
            print("Response content:", response.content)

            response.raise_for_status()  # Raises HTTPError if response is 4xx, 5xx
            return response.json()
        except requests.exceptions.RequestException as e:
            print("Exception occurred during token exchange:", str(e))
            if 'response' in locals():
                print("Error response content:", response.text)
            else:
                print("No response object returned.")
            return {}

    def get_user_info(self, access_token):
        user_info_url = settings.UAE_PASS['USER_INFO_URL']
        headers = {
            'Authorization': f'Bearer {access_token}',
        }
        response = requests.get(user_info_url, headers=headers,proxies={})
        return response.json()
    
    def generate_access_token(self, user):  
        refresh = RefreshToken.for_user(user)
        return {
            'refresh': str(refresh),
            'access': str(refresh.access_token),
        }
    

class SubAuthView(APIView):
    #permission_classes = [permissions.AllowAny , APIKeyPermission]
    permission_classes = [permissions.AllowAny]
    serializer_class = SubAuthSerializer

    def post(self, request, *args, **kwargs):
        serializer = self.serializer_class(data=request.data)
        
        if serializer.is_valid():
            user, tokens, created = serializer.authenticate()
            
            return Response({
                'user': {
                    'id': user.id,
                    'sub': user.sub,
                    'email': user.email,
                    'username': user.username,
                    'user_type': user.user_type,
                    'is_new': created,
                },
                'tokens': tokens
            }, status=status.HTTP_200_OK)
            
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class TokenRefreshView(APIView):
    permission_classes = [permissions.AllowAny, APIKeyPermission]
    
    def post(self, request):
        refresh_token = request.data.get('refresh')
        
        if not refresh_token:
            return Response(
                {'error': 'Refresh token is required'}, 
                status=status.HTTP_400_BAD_REQUEST
            )
        
        try:
            refresh = RefreshToken(refresh_token)
            access_token = str(refresh.access_token)
            
            return Response({
                'access': access_token,
                'refresh': str(refresh),
            }, status=status.HTTP_200_OK)
            
        except TokenError as e:
            return Response(
                {'error': str(e)}, 
                status=status.HTTP_401_UNAUTHORIZED
            )

# done bucket doenload
class AutodeskFileUploadView(APIView):
    """
    Simplified API endpoint for uploading files to Autodesk.
    
    This endpoint automatically:
    1. Creates a bucket if none exists
    2. Uploads the file to that bucket
    3. Returns the bucket key in the response
    
    Usage:
    POST /api/user_management/autodesk/upload/
    Form data:
    - file: The file to upload (required)
    - object_name: Custom name for the file (optional)
    """
    permission_classes = [permissions.AllowAny]
    
    def post(self, request):
        """
        Upload a file to Autodesk bucket.
        """
        try:
            uploaded_file = request.FILES.get('file')
            object_name = request.data.get('object_name')
            
            # Validate required field
            if not uploaded_file:
                return Response({
                    'status': 'error',
                    'message': 'file is required'
                }, status=status.HTTP_400_BAD_REQUEST)
            
            # Save file temporarily
            import tempfile
            import os
            
            # Create a temporary file
            with tempfile.NamedTemporaryFile(delete=False) as tmp_file:
                for chunk in uploaded_file.chunks():
                    tmp_file.write(chunk)
                temp_file_path = tmp_file.name
            
            try:
                # Get access token
                access_token = get_access_token()
                
                # Use the original filename if object_name is not provided
                if not object_name:
                    object_name = uploaded_file.name
                
                # Get buckets
                buckets_data = get_buckets(access_token)
                buckets = buckets_data.get('items', [])
                
                # Use existing bucket or create a new one
                if len(buckets) > 0:
                    # Use the first bucket found
                    bucket_key = buckets[0].get('bucketKey')
                    bucket_message = f"Using existing bucket: {bucket_key}"
                else:
                    # No buckets found, create a new one
                    bucket_key = f"mybucket_{settings.CLIENT_ID[:10].lower()}"
                    bucket_data = create_bucket(access_token, bucket_key, 'transient')
                    bucket_message = f"Created new bucket: {bucket_key}"
                
                # Upload file
                upload_result = upload_file(access_token, bucket_key, temp_file_path, object_name)
                print(f"uploaded result : {upload_result}")
                response_data = {
                    'status': 'success',
                    'message': f'File uploaded successfully. {bucket_message}',
                    'upload_result': {
                        'bucket_key': bucket_key,
                        'object_id': upload_result.get('objectId'),
                        'object_key': upload_result.get('objectKey'),
                        'size': upload_result.get('size'),
                        'download_url': upload_result.get('download_url')
                    }
                }
                
                return Response(response_data, status=status.HTTP_200_OK)
            finally:
                # Clean up temporary file
                if os.path.exists(temp_file_path):
                    os.unlink(temp_file_path)
                    
        except Exception as e:
            return Response({
                'status': 'error',
                'message': f'Error processing request: {str(e)}'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

class AutodeskAccessTokenView(APIView):
    """API endpoint for retrieving Autodesk access token."""
    permission_classes = [permissions.AllowAny]

    def get(self, request):
        try:
            access_token = get_access_token()
            return Response({
                'status': 'success',
                'data': access_token
            }, status=status.HTTP_200_OK)

        except Exception as e:
            return Response({
                'status': 'error',
                'message': f'Failed to retrieve access token: {str(e)}'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

class AutodeskTranslateView(APIView):
    """
    API to translate an Autodesk file to a viewable format (SVF/SVF2).
    
    Usage:
    POST /api/user_management/autodesk/translate/
    Body (JSON):
    {
        "object_id": "urn:adsk.objects:os.object:your_bucket/your_file_name"
    }
    """

    permission_classes = [permissions.AllowAny]

    def post(self, request):
        object_id = request.data.get("object_id")

        if not object_id:
            return Response({
                "status": "error",
                "message": "object_id is required"
            }, status=status.HTTP_400_BAD_REQUEST)

        try:
            # Get a fresh Autodesk access token
            access_token = get_access_token()
            urn = translate_to_viewable(access_token ,object_id )            
            return Response({
                "status": "success",
                "message": "Translation started successfully",
                "urn": urn,   
            }, status=status.HTTP_200_OK)

        except Exception as e:
            return Response({
                "status": "error",
                "message": f"Error processing translation: {str(e)}"
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


# auto desk 3-legged auth
class AutodeskLoginView(APIView):
    """
    Returns the Autodesk OAuth login URL instead of redirecting.
    """
    permission_classes = [permissions.AllowAny]

    def get(self, request):
        callback_url = request.GET.get("callback_url")

        scopes = [
            "data:read", "data:write", "data:create",
            "bucket:read", "bucket:create", "viewables:read",
            "account:read"
        ]

        query = {
            "response_type": "code",
            "client_id": settings.AUTODESK_CLIENT_ID,
            "redirect_uri": callback_url,
            "scope": " ".join(scopes)
        }

        qs = "&".join([f"{k}={v}" for k, v in query.items()])
        login_url = f"{settings.AUTODESK_OAUTH_URL}/authorize?{qs}"

        return Response({
            "status": "success",
            "message": "Open this URL in your browser to log in to Autodesk.",
            "login_url": login_url
        })

class AutodeskCallbackView(APIView):
    permission_classes = [permissions.AllowAny]

    def get(self, request):
        code = request.GET.get("code")
        callback_url = request.GET.get("callback_url")

        if not code:
            return Response({
                "status": "error",
                "message": "Missing authorization code in callback."
            }, status=400)

        token_url = f"{settings.AUTODESK_OAUTH_URL}/token"
        data = {
            "grant_type": "authorization_code",
            "code": code,
            "client_id": settings.AUTODESK_CLIENT_ID,
            "client_secret": settings.AUTODESK_CLIENT_SECRET,
            "redirect_uri": callback_url
        }

        resp = requests.post(token_url, data=data)
        token_data = resp.json()

        return Response({
            "status": "success" if "access_token" in token_data else "error",
            "data": token_data
        })
        
@csrf_exempt
@require_http_methods(["POST"])
def upload_to_acc(request):
    """
    Django endpoint to upload large files to Autodesk ACC (BIM 360 Docs)

    Expected form data:
    - file: The file to upload (multipart/form-data)
    - access_token: ACC access token
    - (optional) project_id: ACC project ID
    - (optional) folder_id: ACC folder ID

    If project_id or folder_id are not provided, the first available ones
    from the user’s hubs will be used automatically.
    """
    try:
        # Extract parameters
        uploaded_file = request.FILES.get('file')
        access_token = request.POST.get('access_token')
        project_id = request.POST.get('project_id')
        folder_id = request.POST.get('folder_id')

        if not uploaded_file:
            return JsonResponse({'status': 'error', 'message': 'No file provided'}, status=400)

        if not access_token:
            return JsonResponse({'status': 'error', 'message': 'Missing access_token'}, status=400)

        file_name = uploaded_file.name
        file_size = uploaded_file.size
        print(f"Uploading {file_name} ({file_size / (1024 * 1024):.2f} MB)")

        # Save uploaded file to temporary path
        if hasattr(uploaded_file, 'temporary_file_path'):
            temp_file_path = uploaded_file.temporary_file_path()
        else:
            temp_file = tempfile.NamedTemporaryFile(delete=False)
            for chunk in uploaded_file.chunks():
                temp_file.write(chunk)
            temp_file.close()
            temp_file_path = temp_file.name

        try:
            # Step 1: Create storage location
            print("Creating storage...")
            storage_id = create_storage_location(access_token, project_id, folder_id, file_name)

            # Step 2: Upload to storage
            print("Uploading file to storage...")
            upload_large_file_to_storage(access_token, storage_id, temp_file_path)

            # Step 3: Create first version
            print("Creating item version in ACC...")
            item_data = create_first_version(access_token, project_id, folder_id, storage_id, file_name)
            print(f"item data : {item_data}")
            print(f"item data included: {item_data['included']}")
            version_id = item_data['included'][0]['id']
            translate_res =  translate_file(access_token,version_id)
            
            # version_id = item_data[""]
            return JsonResponse({
                'status': 'success',
                'message': 'File uploaded successfully',
                'data': {
                    'file_name': file_name,
                    'storage_id': storage_id,
                    'item_id': item_data['data']['id'],
                    'project_id': project_id,
                    'folder_id': folder_id,
                    
                },
                "translate_response" : translate_res,
            }, status=200)

        finally:
            # Ensure file is closed and safely deleted
            if temp_file_path and os.path.exists(temp_file_path):
                try:
                    # Try to close any lingering open handles
                    try:
                        uploaded_file.close()
                    except Exception:
                        pass

                    # Wait briefly to release file lock (Windows-specific)
                    time.sleep(0.5)

                    os.remove(temp_file_path)
                except Exception as cleanup_err:
                    print(f"Cleanup warning: {cleanup_err}")

    except Exception as e:
        print(f"Upload error: {str(e)}")
        return JsonResponse({'status': 'error', 'message': str(e)}, status=500)



