import os
import base64,json , math , time 
from urllib.parse import quote
import requests
from urllib.parse import urlencode
from django.conf import settings

def get_access_token():
    # Read credentials from environment variables (recommended)
    CLIENT_ID = settings.CLIENT_ID
    CLIENT_SECRET = settings.CLIENT_SECRET

    if not CLIENT_ID or not CLIENT_SECRET:
        raise SystemExit("Set APS_CLIENT_ID and APS_CLIENT_SECRET environment variables first.")

    # Endpoint for APS / OAuth v2 token (two-legged client_credentials)
    TOKEN_URL = "https://developer.api.autodesk.com/authentication/v2/token"

    # Choose scopes your app needs (example) -- update to the scopes your app requires
    # IMPORTANT: For listing hubs, you need:
    # 1. "account:read" scope - required to access hubs/projects
    # 2. 3-legged OAuth (user authentication) - 2-legged won't work for hubs!
    SCOPES = [
        "data:read",
        "data:write",
        "bucket:create",
        "bucket:read",
        "account:read",  # Required for accessing hubs
        "viewables:read"  # Required for viewing models in Autodesk Viewer
    ]
    scope_str = " ".join(SCOPES)

    # Build Basic auth header: Base64(client_id:client_secret)
    auth_raw = f"{CLIENT_ID}:{CLIENT_SECRET}".encode("utf-8")
    auth_b64 = base64.b64encode(auth_raw).decode("utf-8")
    headers = {
        "Authorization": f"Basic {auth_b64}",
        "Content-Type": "application/x-www-form-urlencoded"
    }

    # Body for client_credentials grant
    data = {
        "grant_type": "client_credentials",
        "scope": scope_str
    }

    resp = requests.post(TOKEN_URL, headers=headers, data=urlencode(data))
    if resp.status_code != 200:
        raise Exception(f"Failed to get token: {resp.status_code} - {resp.text}")

    token_json = resp.json()
    access_token = token_json.get("access_token")
    expires_in = token_json.get("expires_in")

    return access_token
    
def get_buckets(access_token, region="US"):
    """
    Get all buckets accessible with the current token

    This works with 2-legged OAuth and requires bucket:read scope.
    Region can be "US" or "EMEA"
    """
    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    buckets_url = "https://developer.api.autodesk.com/oss/v2/buckets"

    # Add region parameter if needed
    params = {}
    if region:
        params["region"] = region

    response = requests.get(buckets_url, headers=api_headers, params=params)

    if response.status_code != 200:
        print(f"Failed to get buckets: {response.status_code}")
        print(f"Response: {response.text}")
        response.raise_for_status()

    buckets_data = response.json()
    return buckets_data

def create_bucket(access_token, bucket_key, policy_key="transient"):
    """
    Create a new bucket

    policy_key options:
    - "transient" (24 hour retention)
    - "temporary" (30 day retention)
    - "persistent" (until deleted)
    """
    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    buckets_url = "https://developer.api.autodesk.com/oss/v2/buckets"

    payload = {
        "bucketKey": bucket_key,
        "policyKey": policy_key
    }

    response = requests.post(buckets_url, headers=api_headers, json=payload)

    if response.status_code == 409:
        print(f"Bucket '{bucket_key}' already exists")
        return {"bucketKey": bucket_key, "exists": True}
    elif response.status_code not in [200, 201]:
        print(f"Failed to create bucket: {response.status_code}")
        print(f"Response: {response.text}")
        response.raise_for_status()

    bucket_data = response.json()
    return bucket_data

def translate_to_viewable(access_token, object_id):
    """
    Send a file in Autodesk OSS bucket for translation into viewable format (SVF2).
    """

    # Encode object_id to base64 URN (without padding)
    urn = base64.b64encode(object_id.encode()).decode().rstrip("=")

    # Extract file name from object_id (after the last '/')
    file_name = os.path.basename(object_id)

    url = "https://developer.api.autodesk.com/modelderivative/v1/designdata/job"
    headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    data = {
        "input": {
            "urn": urn,
            "compressedUrn": False,
            "rootFilename": file_name   # ✅ Fix: Set the file name here
        },
        "output": {
            "formats": [
                {
                    "type": "svf2",
                    "views": ["2d", "3d"]
                }
            ]
        }
    }

    # Step 1: Submit translation job
    resp = requests.post(url, headers=headers, data=json.dumps(data))
    if resp.status_code not in [200, 201]:
        return {
            "status": "error",
            "message": f"Translation request failed: {resp.status_code}",
            "response": resp.text
        }

    job_result = resp.json()

    # Step 2: Poll manifest for progress (optional)
    manifest_url = f"https://developer.api.autodesk.com/modelderivative/v2/designdata/{urn}/manifest"
    for _ in range(10):
        manifest_resp = requests.get(manifest_url, headers=headers)
        if manifest_resp.status_code == 200:
            manifest_data = manifest_resp.json()
            status = manifest_data.get("status", "").lower()

            if status == "success":
                return {
                    "status": "success",
                    "message": "File translated successfully.",
                    "urn": urn,
                    "manifest": manifest_data
                }
            elif status == "failed":
                return {
                    "status": "error",
                    "message": "File translation failed.",
                    "manifest": manifest_data
                }

        time.sleep(5)

    return {
        "status": "pending",
        "message": "Translation job submitted but not yet complete.",
        "urn": urn,
        "job_result": job_result
    }

def translate_to_svf(access_token, object_id_or_urn):
    """
    Translate an uploaded file to SVF2 format for viewing in Autodesk Viewer

    Args:
        access_token: OAuth token
        object_id_or_urn: Object ID (like urn:adsk.objects:os.object:...) or base64 URN

    Returns:
        Translation job information and the base64 URN
    """

    # If it starts with 'urn:', it's an object ID that needs to be base64 encoded
    if object_id_or_urn.startswith('urn:'):
        # Base64 encode the URN
        urn_bytes = object_id_or_urn.encode('utf-8')
        safe_urn = base64.urlsafe_b64encode(urn_bytes).decode('utf-8').rstrip('=')
    else:
        # Already base64 encoded
        safe_urn = object_id_or_urn

    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    # Translation endpoint
    translate_url = "https://developer.api.autodesk.com/modelderivative/v2/designdata/job"

    # Request SVF2 format (modern format, better than SVF)
    payload = {
        "input": {
            "urn": safe_urn
        },
        "output": {
            "formats": [
                {
                    "type": "svf2",
                    "views": ["2d", "3d"]
                }
            ]
        }
    }

    response = requests.post(translate_url, headers=api_headers, json=payload)

    if response.status_code not in [200, 201]:
        print(f"Failed to start translation: {response.status_code}")
        print(f"Response: {response.text}")
        response.raise_for_status()

    result = response.json()
    print(f"Translation job started!")
    print(f"URN: {result.get('urn')}")

    # Return both the result and the base64 URN for convenience
    result['base64_urn'] = safe_urn
    return result

def upload_file(access_token, bucket_key, file_path, object_name=None):
    """
    Upload a file to a bucket using the new Direct-to-S3 method with multipart support
    FIXED VERSION: Handles large files without URL expiration issues

    Args:
        access_token: OAuth token
        bucket_key: The bucket key to upload to
        file_path: Local path to the file
        object_name: Name for the object in the bucket (defaults to filename)

    Note: Handles both small files (<5MB) and large files (>=5MB) with multipart upload
    """
    if not os.path.exists(file_path):
        raise FileNotFoundError(f"File not found: {file_path}")

    if object_name is None:
        object_name = os.path.basename(file_path)

    # Get file size
    file_size = os.path.getsize(file_path)

    # Each part must be at least 5MB except the last part
    PART_SIZE = 5 * 1024 * 1024  # 5MB in bytes

    # Calculate number of parts needed
    num_parts = math.ceil(file_size / PART_SIZE) if file_size > PART_SIZE else 1

    print(f"  File size: {file_size:,} bytes ({file_size / (1024 * 1024):.2f} MB)")
    print(f"  Upload strategy: {'Multipart' if num_parts > 1 else 'Single part'} ({num_parts} part{'s' if num_parts > 1 else ''})")

    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    signed_url_endpoint = f"https://developer.api.autodesk.com/oss/v2/buckets/{bucket_key}/objects/{object_name}/signeds3upload"

    # Collect all ETags for each part
    etags = []
    upload_key = None
    upload_urls = []

    # Process file in chunks
    with open(file_path, 'rb') as file:
        for part_index in range(num_parts):
            # FIXED: Request new URLs more frequently to avoid expiration
            # Signed URLs expire after 2 minutes by default
            # We request in batches of 5 with 10-minute expiration to ensure they don't expire
            batch_size_limit = 5  # Request max 5 URLs at a time (safer for timing)

            # Request new URLs every 5 parts OR at the start
            if part_index % batch_size_limit == 0:
                batch_start = part_index + 1
                batch_size = min(batch_size_limit, num_parts - part_index)

                print(f"  Step 1: Getting signed S3 upload URLs for parts {batch_start}-{batch_start + batch_size - 1}...")
                params = {
                    "firstPart": batch_start,
                    "parts": batch_size,
                    "minutesExpiration": 10  # FIXED: Request longer expiration (max 60 minutes)
                }

                if upload_key:
                    params["uploadKey"] = upload_key

                response = requests.get(signed_url_endpoint, headers=api_headers, params=params)

                if response.status_code not in [200, 201]:
                    print(f"  Failed to get signed URL: {response.status_code}")
                    print(f"  Response: {response.text}")
                    response.raise_for_status()

                signed_data = response.json()
                upload_key = signed_data.get('uploadKey')
                upload_urls = signed_data.get('urls', [])

                if not upload_urls:
                    raise Exception("No upload URLs received")

                print(f"  Got {len(upload_urls)} signed URL(s) (valid for 10 minutes)")

            # Get the URL for this specific part
            url_index = part_index % batch_size_limit
            s3_url = upload_urls[url_index]

            # Read the chunk for this part
            chunk = file.read(PART_SIZE)
            chunk_size = len(chunk)

            # Step 2: Upload this part to S3 with retry logic
            print(f"  Step 2: Uploading part {part_index + 1}/{num_parts} ({chunk_size:,} bytes)...")

            # FIXED: Add retry logic for upload with exponential backoff
            max_retries = 3
            for retry in range(max_retries):
                try:
                    s3_response = requests.put(
                        s3_url,
                        data=chunk,
                        headers={"Content-Type": "application/octet-stream"},
                        timeout=300  # FIXED: 5 minute timeout for upload
                    )

                    if s3_response.status_code not in [200, 201]:
                        # FIXED: Handle 403 (expired URL) specifically
                        if s3_response.status_code == 403 and retry < max_retries - 1:
                            # URL might have expired, request a new one
                            print(f"    URL expired, requesting fresh URL...")
                            params = {
                                "firstPart": part_index + 1,
                                "parts": 1,
                                "uploadKey": upload_key,
                                "minutesExpiration": 10
                            }
                            response = requests.get(signed_url_endpoint, headers=api_headers, params=params)
                            if response.status_code in [200, 201]:
                                signed_data = response.json()
                                s3_url = signed_data.get('urls', [])[0]
                                upload_urls[url_index] = s3_url
                                print(f"    Got fresh URL, retrying upload...")
                                time.sleep(1)
                                continue

                        print(f"  Failed to upload part {part_index + 1} to S3: {s3_response.status_code}")
                        print(f"  Response: {s3_response.text}")
                        s3_response.raise_for_status()

                    # Success - get ETag and break retry loop
                    etag = s3_response.headers.get('ETag', '').strip('"')
                    etags.append(etag)
                    print(f"  Part {part_index + 1}/{num_parts} uploaded successfully (ETag: {etag})")
                    break  # Success, exit retry loop

                except Exception as e:
                    if retry < max_retries - 1:
                        wait_time = 2 ** retry  # FIXED: Exponential backoff: 1s, 2s, 4s
                        print(f"    Upload failed: {e}")
                        print(f"    Retrying in {wait_time}s... (attempt {retry + 2}/{max_retries})")
                        time.sleep(wait_time)
                    else:
                        # Final attempt failed, re-raise the exception
                        raise

    # Step 3: Complete the upload
    # Step 3: Complete the upload
    print(f"  Step 3: Completing upload with {len(etags)} part(s)...")
    complete_payload = {
        "uploadKey": upload_key,
        "size": file_size,
        "eTags": etags
    }

    complete_response = requests.post(
        signed_url_endpoint,
        headers=api_headers,
        json=complete_payload
    )

    if complete_response.status_code not in [200, 201]:
        print(f"  ❌ Failed to complete upload: {complete_response.status_code}")
        print(f"  Response: {complete_response.text}")
        raise Exception(f"Failed to complete upload: {complete_response.text}")

    upload_data = complete_response.json() or {}
    print("Response Data:", upload_data)

    print(f"\n✅ Successfully uploaded '{file_path}' as '{object_name}'")
    print(f"Object ID: {upload_data.get('objectId')}")
    print(f"Object Key: {upload_data.get('objectKey')}")
    print(f"Size: {upload_data.get('size', 0):,} bytes")

    # Step 4: Try generating a signed download URL
    print(f"\n  Generating signed download URL...")
    print(f"\n  Generating signed download URL (new API)...")
        # Step 1: Verify the object exists first
    print(f"\n🔍 Verifying object exists...")
    print(f"  Bucket: {bucket_key}")
    print(f"  Object: {object_name}")

    # Check object details endpoint
    details_endpoint = f"https://developer.api.autodesk.com/oss/v2/buckets/{bucket_key}/objects/{quote(object_name, safe='')}/details"

    try:
        details_response = requests.get(details_endpoint, headers=api_headers)
        print(f"  Object details status: {details_response.status_code}")
        
        if details_response.status_code == 200:
            details = details_response.json()
            print(f"  ✅ Object exists: {details.get('objectKey')}")
            print(f"  Size: {details.get('size')} bytes")
        else:
            print(f"  ❌ Object not found: {details_response.text}")
            return upload_data  # Exit early if object doesn't exist
            
    except Exception as e:
        print(f"  ⚠️ Error checking object: {e}")

    # Step 2: Generate signed download URL
    print(f"\n🔗 Generating signed download URL...")

    object_name_encoded = quote(object_name, safe='')
    signed_download_endpoint = f"https://developer.api.autodesk.com/oss/v2/buckets/{bucket_key}/objects/{object_name_encoded}/signed"

    signed_url = None
    try:
        download_payload = {
            "minutesExpiration": 60
        }
        
        download_params = {
            "access": "read"
        }
        
        print(f"  Endpoint: {signed_download_endpoint}")
        
        download_response = requests.post(
            signed_download_endpoint, 
            headers=api_headers, 
            params=download_params,
            json=download_payload
        )

        print(f"  Status: {download_response.status_code}")
        
        if download_response.status_code in [200, 201]:
            download_data = download_response.json()
            signed_url = download_data.get('signedUrl')
            if signed_url:
                print(f"  ✅ Signed download URL generated (valid for 60 minutes)")
                print(f"  URL: {signed_url[:100]}...")
            else:
                print(f"  ⚠️ Response: {download_data}")
        else:
            print(f"  ❌ Error: {download_response.text}")

    except Exception as e:
        print(f"  ⚠️ Exception: {e}")

    # Step 3: Fallback to direct URL
    if not signed_url:
        direct_url = upload_data.get('location')
        if direct_url:
            print(f"\n  📋 Using direct URL (requires Authorization header)")
            print(f"  URL: {direct_url}")
            signed_url = direct_url
        else:
            print(f"  ❌ No URL available")

    upload_data['download_url'] = signed_url

    return upload_data


def create_storage_location(access_token, project_id, folder_id, file_name):
    """
    Step 1: Create storage location in ACC
    """
    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/vnd.api+json"
    }

    storage_url = f"https://developer.api.autodesk.com/data/v1/projects/{project_id}/storage"

    payload = {
        "jsonapi": {"version": "1.0"},
        "data": {
            "type": "objects",
            "attributes": {
                "name": file_name
            },
            "relationships": {
                "target": {
                    "data": {
                        "type": "folders",
                        "id": folder_id
                    }
                }
            }
        }
    }

    response = requests.post(storage_url, headers=api_headers, json=payload)

    if response.status_code not in [200, 201]:
        raise Exception(f"Failed to create storage: {response.status_code} - {response.text}")

    storage_data = response.json()
    storage_id = storage_data['data']['id']
    return storage_id


def translate_file(access_token,version_id):
    """
    Translate a file for viewing in Autodesk Viewer

    POST JSON body:
    {
        "access_token": "your_token",
        "version_id": "urn:adsk.wipprod:fs.file:vf.xxx",  // From upload response
        "wait_for_completion": true  // Optional: wait for translation to complete
    }

    Returns:
    {
        "status": "success",
        "translation": {
            "status": "pending|inprogress|success|failed",
            "urn": "base64_encoded_urn",
            "progress": "complete",
            "viewer_urn": "base64_urn_for_viewer"
        }
    }
    """
    wait_for_completion = False
       

        # Convert version_id to base64 URN
    version_urn_base64 = base64.urlsafe_b64encode(version_id.encode()).decode().rstrip('=')

        # Start translation
    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    translate_url = "https://developer.api.autodesk.com/modelderivative/v2/designdata/job"

    payload = {
        "input": {
            "urn": version_urn_base64
        },
        "output": {
            "formats": [
                {
                    "type": "svf",
                    "views": ["2d", "3d"]
                }
            ]
        }
    }

    response = requests.post(translate_url, headers=api_headers, json=payload)

    if response.status_code not in [200, 201]:
        return{
            'status': 'error',
            'message': f'Translation failed: {response.text}'
        }

    result = response.json()

    response_data = {
        'urn': result.get('urn'),
        'viewer_urn': version_urn_base64,
        'message': 'Translation job started'
    }

    # If wait_for_completion, poll until done
    if wait_for_completion:
        print("Waiting for translation to complete...")
        manifest = wait_for_translation(access_token, version_urn_base64, max_minutes=10)

        if manifest and manifest.get('status') == 'success':
            response_data['translation']['status'] = 'success'
            response_data['translation']['progress'] = 'complete'
            response_data['translation']['message'] = 'Translation completed successfully'
        elif manifest and manifest.get('status') == 'failed':
            response_data['translation']['status'] = 'failed'
            response_data['translation']['message'] = 'Translation failed'
        else:
            response_data['translation']['status'] = 'timeout'
            response_data['translation']['message'] = 'Translation timed out. Check status later.'

    return response_data


def upload_large_file_to_storage(access_token, storage_id, file_path, progress_callback=None):
    """
    Step 2: Upload large file to ACC storage using Direct-to-S3 with multipart

    Args:
        access_token: OAuth token
        storage_id: Storage URN from create_storage_location
        file_path: Path to the file to upload
        progress_callback: Function to call with progress updates (optional)
    """
    if not os.path.exists(file_path):
        raise FileNotFoundError(f"File not found: {file_path}")

    file_size = os.path.getsize(file_path)
    PART_SIZE = 5 * 1024 * 1024  # 5MB chunks
    num_parts = math.ceil(file_size / PART_SIZE) if file_size > PART_SIZE else 1

    # Extract bucket and object key from storage ID
    object_key = storage_id.replace('urn:adsk.objects:os.object:', '')
    parts = object_key.split('/', 1)
    bucket_key = parts[0]
    object_name = parts[1] if len(parts) > 1 else ''

    api_headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/json"
    }

    signed_url_endpoint = f"https://developer.api.autodesk.com/oss/v2/buckets/{bucket_key}/objects/{object_name}/signeds3upload"

    etags = []
    upload_key = None
    upload_urls = []

    # Upload in chunks with progress tracking
    with open(file_path, 'rb') as file:
        for part_index in range(num_parts):
            batch_size_limit = 5  # Request 5 URLs at a time

            # Request new signed URLs every 5 parts
            if part_index % batch_size_limit == 0:
                batch_start = part_index + 1
                batch_size = min(batch_size_limit, num_parts - part_index)

                params = {
                    "firstPart": batch_start,
                    "parts": batch_size,
                    "minutesExpiration": 10  # 10 minute expiration
                }

                if upload_key:
                    params["uploadKey"] = upload_key

                response = requests.get(signed_url_endpoint, headers=api_headers, params=params)

                if response.status_code not in [200, 201]:
                    raise Exception(f"Failed to get signed URL: {response.status_code} - {response.text}")

                signed_data = response.json()
                upload_key = signed_data.get('uploadKey')
                upload_urls = signed_data.get('urls', [])

                if not upload_urls:
                    raise Exception("No upload URLs received")

            # Upload this part
            url_index = part_index % batch_size_limit
            s3_url = upload_urls[url_index]

            chunk = file.read(PART_SIZE)
            chunk_size = len(chunk)

            # Retry logic with exponential backoff
            max_retries = 3
            for retry in range(max_retries):
                try:
                    s3_response = requests.put(
                        s3_url,
                        data=chunk,
                        headers={"Content-Type": "application/octet-stream"},
                        timeout=300  # 5 minute timeout
                    )

                    if s3_response.status_code not in [200, 201]:
                        # Handle expired URL
                        if s3_response.status_code == 403 and retry < max_retries - 1:
                            # Request fresh URL
                            params = {
                                "firstPart": part_index + 1,
                                "parts": 1,
                                "uploadKey": upload_key,
                                "minutesExpiration": 10
                            }
                            response = requests.get(signed_url_endpoint, headers=api_headers, params=params)
                            if response.status_code in [200, 201]:
                                signed_data = response.json()
                                s3_url = signed_data.get('urls', [])[0]
                                upload_urls[url_index] = s3_url
                                time.sleep(1)
                                continue

                        raise Exception(f"Failed to upload part {part_index + 1}: {s3_response.status_code}")

                    # Success - get ETag
                    etag = s3_response.headers.get('ETag', '').strip('"')
                    etags.append(etag)

                    # Call progress callback
                    if progress_callback:
                        progress = ((part_index + 1) / num_parts) * 100
                        progress_callback(progress, part_index + 1, num_parts)

                    break  # Success, exit retry loop

                except Exception as e:
                    if retry < max_retries - 1:
                        wait_time = 2 ** retry  # Exponential backoff
                        time.sleep(wait_time)
                    else:
                        raise

    # Complete the upload
    complete_payload = {
        "uploadKey": upload_key,
        "size": file_size,
        "eTags": etags
    }

    complete_response = requests.post(
        signed_url_endpoint,
        headers=api_headers,
        json=complete_payload
    )

    if complete_response.status_code not in [200, 201]:
        raise Exception(f"Failed to complete upload: {complete_response.status_code} - {complete_response.text}")

    return complete_response.json()

def create_first_version(access_token, project_id, folder_id, storage_id, file_name):
    """
    Create a new item + first version in Autodesk Docs folder.
    """
    url = f"https://developer.api.autodesk.com/data/v1/projects/{project_id}/items"
    headers = {
        "Authorization": f"Bearer {access_token}",
        "Content-Type": "application/vnd.api+json"
    }

    payload = {
        "data": {
            "type": "items",
            "attributes": {
                "displayName": file_name,
                "extension": {
                    "type": "items:autodesk.bim360:File",
                    "version": "1.0"
                }
            },
            "relationships": {
                "tip": {
                    "data": {
                        "type": "versions",
                        "id": "1"
                    }
                },
                "parent": {
                    "data": {
                        "type": "folders",
                        "id": folder_id
                    }
                }
            }
        },
        "included": [
            {
                "type": "versions",
                "id": "1",
                "attributes": {
                    "name": file_name,
                    "extension": {
                        "type": "versions:autodesk.bim360:File",
                        "version": "1.0"
                    }
                },
                "relationships": {
                    "storage": {
                        "data": {
                            "type": "objects",
                            "id": storage_id
                        }
                    }
                }
            }
        ]
    }

    response = requests.post(url, headers=headers, json=payload)
    if response.status_code not in [200, 201]:
        raise Exception(f"Failed to create first version: {response.text}")

    return response.json()

def wait_for_translation(access_token, urn, max_minutes=10):
    """
    Helper function to poll translation status until complete
    """
    max_attempts = max_minutes * 12  # Check every 5 seconds
    attempt = 0

    while attempt < max_attempts:
        attempt += 1
        time.sleep(5)

        try:
            api_headers = {
                "Authorization": f"Bearer {access_token}"
            }

            manifest_url = f"https://developer.api.autodesk.com/modelderivative/v2/designdata/{urn}/manifest"
            response = requests.get(manifest_url, headers=api_headers)

            if response.status_code not in [200, 201]:
                continue

            manifest = response.json()
            status = manifest.get('status')
            progress = manifest.get('progress', 'N/A')

            print(f"Translation progress: {status} - {progress}")

            if status == 'success':
                return manifest
            elif status == 'failed':
                return manifest
            elif status in ['inprogress', 'pending']:
                continue

        except Exception as e:
            print(f"Error checking status: {e}")
            time.sleep(5)
            continue

    return None