from rest_framework import serializers
from rest_framework_simplejwt.tokens import RefreshToken
from .models import User

class UserProfileSerializer(serializers.ModelSerializer):
    user_type_display = serializers.CharField(source='get_user_type_display')
    
    class Meta:
        model = User
        fields = [
            'id', 'sub', 'username', 'email', 'phone_number', 
            'emirates_id', 'user_type', 'user_type_display', 
            'is_active', 'created_at', 'updated_at'
        ]


class UserUpdateSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = ['user_type']

    def update(self, instance, validated_data):
        if validated_data.get('user_type') in [4, 5]:
            instance.is_active = False  

        return super().update(instance, validated_data)
    

class SubAuthSerializer(serializers.Serializer):
    sub = serializers.CharField(required=True)
    email = serializers.EmailField(required=False)
    username = serializers.CharField(required=False)
    
    def get_tokens(self, user):
        refresh = RefreshToken.for_user(user)
        return {
            'refresh': str(refresh),
            'access': str(refresh.access_token),
        }
    
    def authenticate(self):
        validated_data = self.validated_data
        sub = validated_data.get('sub')
        
        try:
            user = User.objects.get(sub=sub)
            
            if 'email' in validated_data and validated_data['email']:
                user.email = validated_data['email']
            if 'username' in validated_data and validated_data['username']:
                user.username = validated_data['username']
                 
            user.save()
                
            return user, self.get_tokens(user), False  
            
        except User.DoesNotExist:
            user_data = {
                'sub': sub,
                'username': validated_data.get('username', f"user_{sub[:8]}"),  
                'email': validated_data.get('email', f"user_{sub[:8]}@admin.com"),
                'user_type': 4,
            }
            
            user = User.objects.create(**user_data)
            return user, self.get_tokens(user), True
    
    
            