from rest_framework import generics
from rest_framework.views import APIView
from django_filters import rest_framework as filters
from openai import OpenAI, AssistantEventHandler
from django.shortcuts import get_object_or_404
from rest_framework.response import Response
from rest_framework import status
from django.conf import settings
from typing_extensions import override
from .models import *
from .serializers import *
from django.db.models import Q
from library.utils import upload_pdf_to_openai, parse_excel_sheet, parse_visual_excel_sheet, parse_excel_simple, parse_visual_excel_simple
from rest_framework.parsers import MultiPartParser, FormParser
from .filters import BookFilter
from rest_framework.permissions import IsAuthenticated , AllowAny
from .permissions import *
import openai
import tempfile
import ezdxf
from random import randint
from .utils import normalize_key
import os


client = OpenAI(api_key=settings.OPENAI_API_KEY)
openai.api_key = settings.OPENAI_API_KEY

class BookFilterView(generics.ListAPIView):
    permission_classes = [APIKeyPermission]
    serializer_class = BookCoverTagSerializer
    filter_backends = (filters.DjangoFilterBackend,)
    filterset_class = BookFilter

    def get_queryset(self):
        return Book.objects.all()

    def filter_queryset(self, queryset):
        filtered_queryset = super().filter_queryset(queryset)
        return filtered_queryset
    
        """ filter by allowed_account_ids """
        # user_type = self.request.user.user_type
        # return [
        #     book for book in filtered_queryset
        #     if not book.allowed_account_ids or user_type in book.allowed_account_ids
        # ]
    


class CollectionListView(generics.ListAPIView):
    permission_classes = [APIKeyPermission]
    queryset = Collection.objects.order_by('order')
    serializer_class = CollectionSerializer

class BookDetailView(generics.RetrieveAPIView):
    permission_classes = [AllowAny, APIKeyPermission]
    queryset = Book.objects.all()
    serializer_class = BookDetailSerializer
    lookup_field = 'id'

        
    def get(self, request, *args, **kwargs):
        # Get the book instance
        book = self.get_object()

        # Check if the user has access to the book
        # user_type = request.user.user_type
        # if book.allowed_account_ids and user_type not in book.allowed_account_ids:
        #     return Response({"error": "You don't have access to this book"}, status=status.HTTP_403_FORBIDDEN)

        # Increment the views count
        book.views += 1
        book.save()

        # Call the original get method to return the response
        return super().get(request, *args, **kwargs)

class CreateNewChat(APIView):
    def post(self, request):
        file_id = request.data.get("file_id")

        if not file_id:
            return Response({"error": "file is not valid"}, status=status.HTTP_400_BAD_REQUEST)

        try:
            thread = openai.beta.threads.create(
                messages=[
                    {
                    "role": "user",
                    "content": "i dont need answers just vector this file",
                    "attachments": [
                        { "file_id": file_id, "tools": [{"type": "file_search"}] }
                    ],
                    }
                ]
            )
            return Response({"chat_id": thread.id}, status=status.HTTP_200_OK)
        
        except Exception as e:
            return Response(
                {"message": "Something went wrong", "error": str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )


class ChatAnswerView(APIView):

    def post(self, request):
        question = request.data.get("question")
        thread_id = request.data.get("thread_id")

        if not question or not thread_id:
            return Response(
                {"error": "Missing 'question' or 'thread_id'."},
                status=status.HTTP_400_BAD_REQUEST,
            )

        try:
            # Send user message to the thread
            client.beta.threads.messages.create(
                thread_id=thread_id,
                role="user",
                content=question,
            )

            # Run the assistant and wait for the response
            run = client.beta.threads.runs.create_and_poll(
                thread_id=thread_id,
                assistant_id=settings.OPENAI_ASSISTANT_ID,
                instructions="Answer as accurately as possible using uploaded documents.",
            )

            # Get the latest assistant message
            messages = client.beta.threads.messages.list(thread_id=thread_id)
            assistant_messages = [m for m in messages.data if m.role == "assistant"]

            if assistant_messages:
                latest_message = assistant_messages[0].content[0].text.value
                return Response({"answer": latest_message})
            else:
                return Response(
                    {"error": "No assistant message found."},
                    status=status.HTTP_500_INTERNAL_SERVER_ERROR,
                )

        except Exception as e:
            return Response({"error": str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
        

class UploadFiles(APIView):
    parser_classes = [MultiPartParser]

    def post(self, request):
        pdf = request.FILES.get("pdf")

        if not pdf:
            return Response({"error": "PDF is required."}, status=400)
        
        # Save files temporarily
        with tempfile.NamedTemporaryFile(suffix=".pdf", delete=False) as f1:
            f1.write(pdf.read())
            file_path = f1.name
            file_id = upload_pdf_to_openai(file_path)
            if file_id:
                return Response({"file_id": file_id}, status=200)
            return Response({"error": "upload failed"}, status=400)


class ComparePDFsView(APIView):

    permission_classes = [IsAuthenticated, UserTypePermission(2)]

    def post(self, request):
        pdf1 = request.data.get("pdf1")
        pdf2 = request.data.get("pdf2")

        if not pdf1 or not pdf2:
            return Response({"error": "Both PDF files are required."}, status=400)

        # Create thread and attach files
        thread = openai.beta.threads.create(
                messages=[
                    {
                    "role": "user",
                    "content": "vectorize this file",
                    # Attach the new file to the message.
                    "attachments": [
                        { "file_id": pdf1, "tools": [{"type": "file_search"}] },
                        { "file_id": pdf2, "tools": [{"type": "file_search"}] }
                    ],
                    }
                ]
            )
        openai.beta.threads.messages.create(
            thread_id=thread.id,
            role="user",
            content="the first pdf is the guideline the second pdf with specs should follow it give me the specs that the second pdf didnt follow return the andwer only and in a peresentation format",
        )

        # Run the assistant
        run = openai.beta.threads.runs.create(
            thread_id=thread.id,
            assistant_id=settings.OPENAI_ASSISTANT_ID,
        )

        # Wait for completion (polling)
        import time
        while True:
            run_status = openai.beta.threads.runs.retrieve(thread_id=thread.id, run_id=run.id)
            if run_status.status == "completed":
                break
            elif run_status.status in ("failed", "cancelled"):
                return Response({"error": f"Run failed with status: {run_status.status}"}, status=500)
            time.sleep(2)

        # Retrieve the response
        messages = openai.beta.threads.messages.list(thread_id=thread.id)
        output = messages.data[0].content[0].text.value

        return Response({"result": output})
    

class GenerateReportView(APIView):

    permission_classes = [IsAuthenticated]

    def get_object(self, pk):
        
        return get_object_or_404(Project, pk=pk)
    
    def get(self, request, pk):
        
        try:
            user = request.user
            if user.user_type not in [4, 5]:
                return Response(
                    {"detail": "You do not have permission to perform this action."},
                    status=status.HTTP_403_FORBIDDEN
                )
            
            project = self.get_object(pk)
            
            if project.owner != user and project.contractor != user:
                return Response(
                    {"detail": "You do not have permission to perform this action."},
                    status=status.HTTP_403_FORBIDDEN
                )
          
            if not project.report:
                return Response(
                    {"error": "Project has no report file attached"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
                
            if not project.visual_report:
                return Response(
                    {"error": "Project has no visual report file attached"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
            
          
            report_data = parse_excel_sheet(project.report.path)
            visual_report_data = parse_visual_excel_sheet(project.visual_report.path)
            
            return Response({
                "report_data": report_data, 
                "visual_report_data": visual_report_data
            })
            
        except FileNotFoundError as e:
            print(f"File not found: {str(e)}")
            return Response(
                {"error": "Report file not found"}, 
                status=status.HTTP_404_NOT_FOUND
            )
        except Exception as e:
            print(f"Failed to generate report: {str(e)}")
            return Response(
                {"error": f"Failed to generate report: {str(e)}"}, 
                status=status.HTTP_400_BAD_REQUEST
            )
            
class PublicGenerateReportView(APIView):
    """
    Public API endpoint for generating reports without authentication.
    Uses simplified Excel parsing functions.
    """
    permission_classes = []
    authentication_classes = []
    
    def get_object(self, pk):
        return get_object_or_404(Project, pk=pk)
    
    def get(self, request, pk):
        try:
            project = self.get_object(pk)
            
            if not project.report:
                return Response(
                    {"error": "Project has no report file attached"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
                
            if not project.visual_report:
                return Response(
                    {"error": "Project has no visual report file attached"}, 
                    status=status.HTTP_404_NOT_FOUND
                )
            
            # Use the new simplified parsing functions
            report_data = parse_excel_simple(project.report.path)
            visual_report_data = parse_visual_excel_simple(project.visual_report.path)
            
            return Response({
                "report_data": report_data, 
                "visual_report_data": visual_report_data,
            })
            
        except FileNotFoundError as e:
            print(f"File not found: {str(e)}")
            return Response(
                {"error": "Report file not found"}, 
                status=status.HTTP_404_NOT_FOUND
            )
        except Exception as e:
            print(f"Failed to generate report: {str(e)}")
            return Response(
                {"error": f"Failed to generate report: {str(e)}"}, 
                status=status.HTTP_400_BAD_REQUEST
            )
            
class GenerateReportByUploadView(APIView):
    permission_classes = [AllowAny, APIKeyPermission]

    def post(self, request):
        try:
            report_file = request.FILES.get('report_data')
            visual_report_file = request.FILES.get('visual_report_data')

            if not report_file or not visual_report_file:
                return Response(
                    {"error": "Both 'report_file' and 'visual_report_file' are required."},
                    status=status.HTTP_400_BAD_REQUEST
                )
                
            report_data = parse_excel_sheet(report_file)
            visual_report_data = parse_visual_excel_sheet(visual_report_file)

            return Response({
                "report_data": report_data,
                "visual_report_data": visual_report_data
            })

        except Exception as e:
            print(f"Failed to generate report: {str(e)}")
            return Response(
                {"error": f"Failed to generate report: {str(e)}"},
                status=status.HTTP_400_BAD_REQUEST
            )
            
class GenerateReportByPathView(APIView):

    permission_classes = [AllowAny, APIKeyPermission]

    def post(self, request):
        
        try:
            report_data_path = request.data.get('report_data')
            visual_report_data_path = request.data.get('visual_report_data')
          
            report_data = parse_excel_sheet(report_data_path)
            visual_report_data = parse_visual_excel_sheet(visual_report_data_path)
            
            return Response({
                "report_data": report_data, 
                "visual_report_data": visual_report_data
            })
        except Exception as e:
            print(f"Failed to generate report: {str(e)}")
            return Response(
                {"error": f"Failed to generate report: {str(e)}"}, 
                status=status.HTTP_400_BAD_REQUEST
            )



class ProjectListView(generics.ListAPIView):
    """
    List all projects where the user is either owner or contractor
    """
    serializer_class = ProjectSerializer
    permission_classes = [IsAuthenticated,UserTypePermission(4,5)]
    
    def get_queryset(self):
        user = self.request.user
        return Project.objects.filter(
            Q(owner=user) | Q(contractor=user)
        ).select_related('owner', 'contractor')
    
class ProjectCreateView(APIView):
    """
    Create a new project with 2 files and project name.
    Automatically assigns current user as both owner and contractor.
    """
    permission_classes = [IsAuthenticated]
    parser_classes = [MultiPartParser]
    
    def post(self, request):
        serializer = ProjectCreateSerializer(data=request.data)
        
        if not serializer.is_valid():
            return Response(
                serializer.errors, 
                status=status.HTTP_400_BAD_REQUEST
            )
        
        # Get validated data
        validated_data = serializer.validated_data
        
        # Add current user as both owner and contractor
        validated_data['owner'] = request.user
        validated_data['contractor'] = request.user
        
        try:
            # Create the project
            project = Project.objects.create(**validated_data)
            
            # Return the created project with detailed info
            response_serializer = ProjectSerializer(project, context={'request': request})
            return Response(
                response_serializer.data,
                status=status.HTTP_201_CREATED
            )
            
        except Exception as e:
            return Response(
                {"error": f"Failed to create project: {str(e)}"}, 
                status=status.HTTP_400_BAD_REQUEST
            )




class SubmissionsByProjectView(APIView):
    '''
    list submissions of a specific project
    '''
    permission_classes = [IsAuthenticated,IsProjectOwnerOrContractor]

    def get(self, request, project_id):
        project = get_object_or_404(Project, id=project_id)
        self.check_object_permissions(request, project)
        submissions = SubmissionRequest.objects.filter(project=project)
        serializer = SubmissionRequestDetailSerializer(submissions, many=True, context={'request': request})
        return Response(serializer.data)
    

class SubmissionRequestCreateView(generics.CreateAPIView):
    '''
    contractor create request
    '''
    serializer_class = SubmissionRequestCreateSerializer
    permission_classes = [IsAuthenticated]
    
    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        submission_request = serializer.save()
        
        # Return the created submission with detailed info
        detail_serializer = SubmissionRequestDetailSerializer(submission_request, context={'request': request})
        return Response(
            detail_serializer.data, 
            status=status.HTTP_201_CREATED
        )


class SubmissionRequestDetailView(generics.RetrieveAPIView):
    """
    Get specific submission request details by ID
    Only accessible by project owner or contractor
    """
    serializer_class = SubmissionRequestDetailSerializer
    permission_classes = [IsAuthenticated, IsSubmissionOwnerOrContractor]
    queryset = SubmissionRequest.objects.select_related('project')
    lookup_field = 'id'
    lookup_url_kwarg = 'id'


class SubmissionListView(generics.ListAPIView):
    """
    List all submission requests for the inspector
    """
    serializer_class = SubmissionRequestDetailSerializer
    permission_classes = [IsAuthenticated,UserTypePermission(2)]

    def get_queryset(self):
        queryset = SubmissionRequest.objects.all()
        status = self.request.query_params.get('status')
        if status:
            queryset = queryset.filter(status=status)
        return queryset

class UpdateSubmissionRestrictedFieldsView(APIView):
    permission_classes = [IsAuthenticated, UserTypePermission(2)]

    def patch(self, request, pk):
        submission = get_object_or_404(SubmissionRequest, pk=pk)

        serializer = SubmissionRestrictedUpdateSerializer(submission, data=request.data, partial=True, context={'request': request})
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    


class CADVerifierAPIView(APIView):
    parser_classes = [MultiPartParser, FormParser]
    
    def post(self, request):
        serializer = CADVerifierSerializer(data=request.data)
        
        if not serializer.is_valid():
            return Response(
                {"error": "Invalid input data", "details": serializer.errors}, 
                status=status.HTTP_400_BAD_REQUEST
            )
        
        verifier_type = serializer.validated_data["verifier_type"]
        uploaded_file = serializer.validated_data["file"]
        
        # Map verifier types to titles
        verifier_titles = {
            "architectural_test": "Architectural Test",
            "mep_test": "MEP Test",
            "structural_test": "Structural Test",
        }
        
        verifier_title = verifier_titles[verifier_type]
        test = VerificationTest.objects.filter(title__icontains=verifier_title).first()
        
        if not test:
            return Response(
                {"error": f"No test found for verifier type '{verifier_type}'"},
                status=status.HTTP_404_NOT_FOUND
            )
        
        # Generate unique filename
        filename = f"CAD_VER_{randint(0, 1000)}.dxf"
        
        try:
            # Save and process the DXF file
            with open(filename, "wb") as fs:
                for chunk in uploaded_file.chunks():
                    fs.write(chunk)
            
            doc = ezdxf.readfile(filename)
            layer_names = [layer.dxf.name for layer in doc.layers]
            
            # Clean up file
            if os.path.exists(filename):
                os.remove(filename)
                
        except Exception as ex:
            # Ensure cleanup on error
            if os.path.exists(filename):
                os.remove(filename)
            
            return Response(
                {"error": "Failed to process DXF file", "details": str(ex)}, 
                status=status.HTTP_400_BAD_REQUEST
            )
        
        # Run tests and prepare results
        results = test.run_test(layer_names)
        normalized_results = {normalize_key(k): v for k, v in results.items()}
        
        num_passing_tests = sum(1 for result in results.values() if result)
        num_failing_tests = sum(1 for result in results.values() if not result)
        total_tests = test.number_of_tests
        valid = num_failing_tests == 0
        
        meta_data = {
            "verifier_type": verifier_type,
            "passing_tests": num_passing_tests,
            "failing_tests": num_failing_tests,
            "total_tests": total_tests,
            "total_layers": len(layer_names),
            "file_name": uploaded_file.name,
        }
        
        response_data = {
            "valid": valid,
            "results": normalized_results,
            "meta_data": meta_data,
        }
        
        return Response(response_data, status=status.HTTP_200_OK)


UUID_MAPPING = {
    "bac2b1fa-9c69-4a8e-9bde-ce11b59f3666": 1, 
    "7d8d2df2-d868-4017-861f-3430c856465b": 2, 
    "86ffd71a-a31f-4f26-8737-98a315c0f895": 3 
}

class SubTestBulkCreateAPIView(APIView):
    permission_classes = [AllowAny]
    def post(self, request):
        data = request.data

        created = []
        errors = []

        for item in data:
            uuid_str = str(item['verificationtest'])
            test_id = UUID_MAPPING.get(uuid_str)

            if not test_id:
                errors.append(f"Unknown verificationtest UUID: {uuid_str}")
                continue

            try:
                vt = VerificationTest.objects.get(pk=test_id)
                subtest = SubTest.objects.create(
                    title=item['title'],
                    description=item.get('description', ''),
                    regex=item['regex'],
                    verificationtest=vt
                )
                created.append(subtest.title)
            except Exception as e:
                errors.append(f"{item['title']}: {str(e)}")

        return Response({
            'created': created,
            'errors': errors
        }, status=status.HTTP_201_CREATED if created else status.HTTP_400_BAD_REQUEST)