from rest_framework import serializers
from .models import *
from library.utils import APIFileURLModelSerializer
from django.core.validators import FileExtensionValidator

class TagSerializer(APIFileURLModelSerializer):
    class Meta:
        model = Tag
        fields = ['title_en', 'title_ar', 'color']

class BookCoverTagSerializer(APIFileURLModelSerializer):
    tag = TagSerializer()

    class Meta:
        model = Book
        fields = ['id', 'cover_img', 'title_en', 'title_ar', 'tag', 'brief_about_en', 
                  'brief_about_ar', 'about_en', 'about_ar' , 'allowed_account_ids']


class CollectionSerializer(APIFileURLModelSerializer):
    class Meta:
        model = Collection
        fields = ['id', 'title_en', 'title_ar']

class LanguageSerializer(APIFileURLModelSerializer):
    class Meta:
        model = Language
        fields = ['lang']

class TargetAudienceSerializer(APIFileURLModelSerializer):
    class Meta:
        model = TargetAudience
        fields = ['title_en', 'title_ar']

class BookDetailSerializer(APIFileURLModelSerializer):
    collection = CollectionSerializer(many=True)
    book_lang = LanguageSerializer(many=True)
    target_audience = TargetAudienceSerializer(many=True)
    tag = TagSerializer()

    class Meta:
        model = Book
        fields = '__all__'


class ProjectSerializer(APIFileURLModelSerializer):
    
    class Meta:
        model = Project
        fields = [
            'id', 'name', 'code', 'cover_img'
        ]
        
class ProjectCreateSerializer(serializers.ModelSerializer):
    report = serializers.FileField(required=True)
    visual_report = serializers.FileField(required=True)
    
    class Meta:
        model = Project
        fields = ['name', 'report', 'visual_report']


class SubmissionSerializer(APIFileURLModelSerializer):
    class Meta:
        model = SubmissionRequest
        fields = '__all__'


class SubmissionRequestCreateSerializer(serializers.ModelSerializer):
    project_id = serializers.IntegerField(write_only=True)
    
    class Meta:
        model = SubmissionRequest
        fields = ['submission_file', 'submission_link', 'project_id']
    
    def validate_project_id(self, value):
        """Validate that the project exists and user has permission"""
        try:
            project = Project.objects.get(id=value)
        except Project.DoesNotExist:
            raise serializers.ValidationError("Project does not exist.")
        
        # Check if user is owner or contractor of the project
        request = self.context.get('request')
        if request and request.user.is_authenticated:
            user = request.user
            if project.contractor != user:
                raise serializers.ValidationError("You don't have permission to create submissions for this project.")
        return value
    
    def create(self, validated_data):
        project_id = validated_data.pop('project_id')
        project = Project.objects.get(id=project_id)
        
        # Create the submission request with the project
        submission_request = SubmissionRequest.objects.create(
            project=project,
            **validated_data
        )
        return submission_request

class SubmissionRequestDetailSerializer(APIFileURLModelSerializer):
    project = serializers.CharField(source='project.name', read_only=True)
    
    class Meta:
        model = SubmissionRequest
        fields = [
            'id', 'status', 'submission_file', 'submission_link',
            'pdf', 'url_360_viewer', 'matterport_link', 'workflow_state', 
            'project', 'created_at', 'updated_at'
        ]


class SubmissionRestrictedUpdateSerializer(APIFileURLModelSerializer):
    class Meta:
        model = SubmissionRequest
        exclude = ['project', 'submission_file', 'submission_link', 'created_at', 'updated_at']


class CADVerifierSerializer(serializers.Serializer):
    file = serializers.FileField(
        validators=[FileExtensionValidator(allowed_extensions=["dxf"])]
    )
    verifier_type = serializers.ChoiceField(
        choices=[
            ("architectural_test", "Architectural Test"),
            ("mep_test", "MEP Test"),
            ("structural_test", "Structural Test"),
        ]
    )