from rest_framework.permissions import BasePermission
from django.conf import settings
from rest_framework.exceptions import APIException
from rest_framework import status

class IsProjectOwnerOrContractor(BasePermission):
    """
    Permission that allows access only to project owners or contractors
    """
    def has_permission(self, request, view):
        return request.user and request.user.is_authenticated
    
    def has_object_permission(self, request, view, obj):
        # Allow access if user is owner or contractor
        return obj.owner == request.user or obj.contractor == request.user

class IsProjectOwner(BasePermission):
    """
    Permission that allows access only to project owners
    """
    def has_permission(self, request, view):
        return request.user and request.user.is_authenticated
    
    def has_object_permission(self, request, view, obj):
        return obj.owner == request.user

class IsProjectContractor(BasePermission):
    """
    Permission that allows access only to project contractors
    """
    def has_permission(self, request, view):
        return request.user and request.user.is_authenticated
    
    def has_object_permission(self, request, view, obj):
        return obj.contractor == request.user

class IsUserType(BasePermission):
    """
    Allows access only to authenticated users with a specific user_type.
    """

    def __init__(self, allowed_types=None):
        self.allowed_types = allowed_types or []

    def has_permission(self, request, view):
        user = request.user
        return user and user.is_authenticated and getattr(user, 'user_type', None) in self.allowed_types
    

def UserTypePermission(*allowed_types):
    class DynamicUserTypePermission(IsUserType):
        def __init__(self):
            super().__init__(allowed_types)
    return DynamicUserTypePermission


class IsSubmissionOwnerOrContractor(BasePermission):
    """
    Permission that allows access only to project owners or contractors
    """
    def has_permission(self, request, view):
        return request.user and request.user.is_authenticated
    
    def has_object_permission(self, request, view, obj):
        # Allow access if user is owner or contractor
        return obj.project.owner == request.user or obj.project.contractor == request.user


class InvalidAPIKey(APIException):
    status_code = status.HTTP_401_UNAUTHORIZED
    default_detail = {
        'status': False,
        'message': 'Unauthorized API.'
    }
    default_code = 'unauthorized'


class APIKeyPermission(BasePermission):
    def has_permission(self, request, view):
        expected_key = getattr(settings, 'API_KEY', None)
        api_key = request.headers.get('X-API-KEY')

        if not expected_key:
            raise APIException('Server misconfigured: missing API key.', code='server_error')

        if api_key != expected_key:
            raise InvalidAPIKey()

        return True