from django.db import models
from django.utils import timezone
from user_management.models import User
from rest_framework.exceptions import ValidationError

# Create your models here.

def default_account_ids():
    return [1, 2, 3, 4, 5, 6]

class Book(models.Model):
     
    DIRECTION_CHOICES = [
        ('LTR', 'Left-to-Right'),
        ('RTL', 'Right-to-Left'),
    ]

    cover_img = models.ImageField(upload_to='book-cover-photo/')
    collection = models.ManyToManyField('Collection',related_name='books_collections')
    title_en = models.CharField(max_length=200)
    title_ar = models.CharField(max_length=200)
    about_en = models.TextField()
    about_ar = models.TextField()
    brief_about_en = models.TextField(blank=True, null=True)
    brief_about_ar = models.TextField(blank=True, null=True)
    page_no = models.IntegerField(default=0)
    webbook = models.URLField(blank=True, null=True)
    epub = models.FileField(upload_to='epub-files/', blank=True, null=True)
    order = models.IntegerField(default=0)
    publish_date = models.DateField(blank=True, null=True)
    book_lang = models.ManyToManyField('Language',related_name='books_langs')
    target_audience = models.ManyToManyField('TargetAudience',related_name='books_TrgAudience')
    direction = models.CharField(
        max_length=3,
        choices=DIRECTION_CHOICES,
        default='LTR',
    )
    author = models.CharField(max_length=200, blank=True, null=True)
    price = models.FloatField(default=0.0)
    views = models.IntegerField(default=0)
    tag = models.ForeignKey('Tag',on_delete=models.SET_NULL,null=True,blank=True,related_name='book_tags')
    allowed_account_ids = models.JSONField(default=default_account_ids)
    created_at = models.DateTimeField(default=timezone.now)
    updated_at = models.DateTimeField(default=timezone.now)
    pdf = models.FileField(upload_to='pdf-files/', blank=True, null=True)
    openai_book_id = models.CharField(max_length=200, blank=True, null=True)

    def __str__(self) -> str:
        return f"{self.title_ar}-{self.title_en}"


class Collection(models.Model):
    title_en = models.CharField(max_length=200)
    title_ar = models.CharField(max_length=200)
    order = models.IntegerField(default=0)

    def __str__(self) -> str:
        return f"{self.title_ar}-{self.title_en}"

class Language(models.Model):
    lang = models.CharField(max_length=100)

    def __str__(self) -> str:
        return self.lang

class TargetAudience(models.Model):
    title_en = models.CharField(max_length=200)
    title_ar = models.CharField(max_length=200) 

    def __str__(self) -> str:
        return f"{self.title_ar}-{self.title_en}"
    
class Tag(models.Model):
    title_en = models.CharField(max_length=200)
    title_ar = models.CharField(max_length=200)
    color =  models.CharField(max_length=200)

    def __str__(self) -> str:
        return f"{self.title_ar}-{self.title_en}"
    

class Project(models.Model):
    name = models.CharField(max_length=200)
    code = models.CharField(max_length=200 ,null=True,blank=True)
    cover_img = models.ImageField(upload_to='project-cover-photo/',null=True,blank=True)
    contractor = models.ForeignKey(User,on_delete=models.SET_NULL,null=True,blank=True,related_name='project_contractor')
    owner = models.ForeignKey(User,on_delete=models.SET_NULL,null=True,blank=True,related_name='project_owner')
    report = models.FileField(upload_to='project-files/', blank=True, null=True)
    visual_report = models.FileField(upload_to='project-files/', blank=True, null=True)

    def clean(self):
        if self.owner and self.contractor and self.contractor.user_type != 4 and self.owner.user_type != 5:
            raise ValidationError('sign a correct contractor and owner')


class SubmissionRequest(models.Model):
    STATUS_CHOICES = [
        (1, 'In Progress'),
        (2, 'Pending'),
        (3, 'Rejected'),
        (4, 'Complete'),
    ]

    WORKFLOW_STATES = [
        (1, 'AI Powered Site Inspection'),
        (2, 'Generative AI PIM'),
        (3, 'Structural Analysis'),
        (4, 'Automated Code'),
        (5, 'Integration'),
    ]

    status = models.IntegerField(choices=STATUS_CHOICES, default=2)
    submission_file = models.FileField(upload_to='submissions-files/', null=True, blank=True)
    submission_link = models.URLField(null=True, blank=True)

    pdf = models.FileField(upload_to='submissions-pdfs/', null=True, blank=True)
    url_360_viewer = models.URLField(null=True, blank=True)
    matterport_link = models.URLField(null=True, blank=True)

    workflow_state = models.IntegerField(choices=WORKFLOW_STATES, default=1)

    project = models.ForeignKey('Project', on_delete=models.CASCADE, related_name='submission_requests')

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"SubmissionRequest #{self.id} - {self.status}"
    

class VerificationTest(models.Model):
    title = models.CharField(max_length=128)
    description = models.TextField(blank=True)

    def __str__(self):
        return self.title

    @property
    def number_of_tests(self):
        return len(self.subtest_set.all())

    def run_test(self, layer_list):
        results = {}

        for subtest in self.subtest_set.all():
            results[subtest.title] = subtest.run_test(layer_list)

        return results


class SubTest(models.Model):
    title = models.CharField(max_length=128)
    description = models.TextField(blank=True)
    regex = models.CharField(max_length=256)
    verificationtest = models.ForeignKey(VerificationTest, on_delete=models.CASCADE)

    def __str__(self):
        return self.title

    def run_test(self, layer_list):
        import re

        reg = re.compile(self.regex)

        for layer in layer_list:
            if reg.match(layer):
                return True

        return False
