from django_filters import rest_framework as filters
from django.db.models.functions import Lower
from django.db.models import Q, Value, BooleanField
from .models import Book

class BookFilter(filters.FilterSet):
    title_en = filters.CharFilter(field_name="title_en", lookup_expr='icontains')
    title_ar = filters.CharFilter(field_name="title_ar", lookup_expr='icontains')
    collection = filters.NumberFilter(method='filter_by_collection')
    book_lang = filters.CharFilter(method='filter_by_book_lang')
    price_gt = filters.NumberFilter(field_name="price", lookup_expr='gt')
    price_e = filters.NumberFilter(field_name="price")
    publish_date_from = filters.DateFilter(field_name="publish_date", lookup_expr='gte')
    publish_date_to = filters.DateFilter(field_name="publish_date", lookup_expr='lte')
    target_audience = filters.CharFilter(method='filter_by_trg_audience')
    sort_by = filters.OrderingFilter(
        fields=(
            ('views', 'views'),
            ('order', 'order'),
            ('publish_date', 'created_at'),
            ('title_en', 'title_en'),
            ('title_ar', 'title_ar')
        )
    )
    
    # Add a search attribute
    search = filters.CharFilter(method='filter_by_search')

    class Meta:
        model = Book
        fields = [
            'title_en',
            'title_ar',
            'collection',
            'book_lang',
            'price',
            'publish_date',
        ]

    def filter_by_collection(self, queryset, name, value):
        # Use Q objects to combine the filters with OR logic
        return queryset.filter(
            Q(collection__id=value) 
        )

    def filter_by_book_lang(self, queryset, name, value):
        return queryset.filter(
            book_lang__lang__icontains=value
        )
    
    def filter_by_search(self, queryset, name, value):
        # Split the input value into separate words
        search_terms = value.split()

        # Step 1: Filter for exact matches of the whole value in any of the fields
        whole_value_query = (
            Q(title_ar__icontains=value) |
            Q(title_en__icontains=value) |
            Q(about_ar__icontains=value) |
            Q(about_en__icontains=value) 
        )

         # Step 2: Build a Q object for individual word matches
        term_query = Q()
        for term in search_terms:
            term_query |= (
                Q(title_ar__icontains=term) |
                Q(title_en__icontains=term) |
                Q(about_ar__icontains=term) |
                Q(about_en__icontains=term) 
            )

        # Step 3: Combine whole-value query with the individual terms query
        combined_query = whole_value_query | term_query

        # Step 4: Annotate the queryset with a boolean flag to prioritize exact matches
        queryset = queryset.annotate(
            is_exact_match=Value(True, output_field=BooleanField())
        ).filter(combined_query).distinct()

        # Step 5: Order by the annotation so that exact matches come first
        return queryset.order_by('-is_exact_match')
    
    def filter_queryset(self, queryset):

        queryset = super().filter_queryset(queryset)
        
        # Get the ordering parameter from the request
        ordering = self.data.get('sort_by')
        if ordering:
            if 'title_en' in ordering:
                # Annotate with case-insensitive field and then order by it
                queryset = queryset.annotate(lower_title_en=Lower('title_en'))
                if ordering.startswith('-'):
                    queryset = queryset.order_by('-lower_title_en')
                else:
                    queryset = queryset.order_by('lower_title_en')
            elif 'title_ar' in ordering:
                # Annotate with case-insensitive field and then order by it
                queryset = queryset.annotate(lower_title_ar=Lower('title_ar'))
                if ordering.startswith('-'):
                    queryset = queryset.order_by('-lower_title_ar')
                else:
                    queryset = queryset.order_by('lower_title_ar')
            else:
                # Fall back to default ordering
                queryset = queryset.order_by(*ordering.split(','))

        return queryset

    def filter_by_trg_audience(self, queryset, name, value):
        return queryset.filter(
            target_audience__title_en__exact=value
        )


