# views.py
from django.db.models import Sum, Count
from django.core.exceptions import PermissionDenied
from django.utils import timezone
from Baldya import settings
from rest_framework import generics, status
from django.shortcuts import get_object_or_404 
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated , AllowAny
from .models import *
from library.models import Book
from library.serializers import BookCoverTagSerializer
from .serializers import CourseSerializer, VideoSerializer, CourseWithVideosSerializer, TempSerializer \
, CustomerHappinessRecordSerializer, CustomerMailSerializer, ContactUsSerializer, CourseCollectionSerializer, VideoQuestionSerializer, MarkVideoWatchedSerializer, CourseQuizSerializer, QuizSubmissionSerializer, CourseEnrollmentSerializer

from django_filters import rest_framework as filters
from .filters import CourseFilter
from .permissions import APIKeyPermission, IsEnrolledInCoursePerm, HasWatchedPreviousVideoPerm, CanAccessCourseQuizPerm

if settings.DEBUG:
    def filter_courses_by_user_type(queryset, user):
        return queryset.extra(
            where=[
                "json_array_length(allowed_account_ids) = 0 OR "
                "json_extract(allowed_account_ids, '$') LIKE %s"
            ],
            params=[f'%{user.user_type}%']
        )
else:
    def filter_courses_by_user_type(queryset, user):
        return queryset.filter(
            Q(allowed_account_ids__len=0) |
            Q(allowed_account_ids__contains=[user.user_type])
        )
        
def is_anonymous(user):
    return not user.is_authenticated


class CourseListView(generics.ListAPIView):
    serializer_class = CourseSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        queryset = Course.objects.order_by('order')
        user = self.request.user
        return queryset

        # ------ filter by allowed_account_ids ----
        # if is_anonymous(user):
        #     return queryset
        # return filter_courses_by_user_type(queryset, user)

class UserRegisteredCoursesView(generics.ListAPIView):
    """
    Get all courses the user registered in
    """
    serializer_class = CourseSerializer
    permission_classes = [IsAuthenticated]
    
    def get_queryset(self):
        return self.request.user.user_registered_courses.order_by('order')

class CourseVideosView(APIView):
    permission_classes = [APIKeyPermission]
    def get(self, request, id):
        course = get_object_or_404(Course, id=id)

        #  Increment views for all requests
        course.views += 1
        course.save(update_fields=["views"])

        # Serialize course with videos
        serializer = CourseWithVideosSerializer(course, context={"request": request})
        data = serializer.data
        videos_data = data.get("videos", [])

        user = request.user
        is_enrolled = (
            user.is_authenticated
            and UserCourse.objects.filter(user=user, course=course).exists()
        )

        # Hide all links except the first
        for video in videos_data[1:]:
            video["link"] = None
            video["transcript_ar"] = None
            video["transcript_en"] = None

        # Anonymous / not enrolled → no watched videos and no course progress
        if not is_enrolled:
            data["watched_videos"] = []
            data["course_progress"] = None
        # Else (enrolled + auth) → watched_videos and course_progress already populated by serializer

        return Response(data, status=status.HTTP_200_OK)


class VideoDetailView(generics.RetrieveAPIView):
    permission_classes = [AllowAny , APIKeyPermission]
    queryset = Video.objects.all()
    serializer_class = VideoSerializer
    lookup_field = 'id'

    def get_object(self):
        video = super().get_object()
        
        
        # ------ Check course access and filter by allowed_account_ids ----
        # course = video.course
        # user = self.request.user
        # if not is_anonymous(user):
        #     if course.allowed_account_ids and user.user_type not in course.allowed_account_ids:
        #         raise PermissionDenied("You don't have access to this video's course")
        
        return video


class VideoQuestionAPIView(generics.RetrieveAPIView):
    """
    API to retrieve video with questions. Requires user to be enrolled in course 
    and have watched previous video (except for first video).
    """
    permission_classes = [IsEnrolledInCoursePerm, HasWatchedPreviousVideoPerm]
    queryset = Video.objects.all()
    serializer_class = VideoQuestionSerializer
    lookup_field = 'pk'
    lookup_url_kwarg = 'video_id'

class CourseSearchAPIView(generics.ListAPIView):
    permission_classes = [APIKeyPermission]
    serializer_class = CourseSerializer
    filter_backends = (filters.DjangoFilterBackend,)
    filterset_class = CourseFilter

    def get_queryset(self):
        queryset = Course.objects.all().order_by('id')
        filtered_queryset = self.filter_queryset(queryset)
        return filtered_queryset
    
    def list(self, request, *args, **kwargs):
        """Override list method to handle course progress enrollment check"""
        queryset = self.filter_queryset(self.get_queryset())
        serializer = self.get_serializer(queryset, many=True)
        serialized_data = serializer.data
        
        # Process course progress based on enrollment
        self._process_course_progress(serialized_data, request)
        
        return Response(serialized_data)
    
    def _process_course_progress(self, serialized_data, request):
        """Process course progress field based on user enrollment"""
        user = request.user
        
        # If user is not authenticated, set all course_progress to None
        if not user.is_authenticated:
            for course_data in serialized_data:
                course_data['course_progress'] = None
            return
        
        # Get enrolled course IDs for the user
        enrolled_course_ids = set(
            UserCourse.objects.filter(user=user).values_list('course_id', flat=True)
        )
        
        # Set course_progress to None for non-enrolled courses
        for course_data in serialized_data:
            if course_data['id'] not in enrolled_course_ids:
                course_data['course_progress'] = None
    
        # ------ Check course access and filter by allowed_account_ids ----
        # user = self.request.user
        # if is_anonymous(user):
        #     return filtered_queryset
        # return filter_courses_by_user_type(filtered_queryset, user)
    
class TempListView(generics.ListAPIView):
    queryset = Temporary.objects.order_by('order')
    serializer_class = TempSerializer

class HomePageApi(APIView):
    def get(self, request, *args, **kwargs):
        # Get the first 10 records from ModelA
        courses = Course.objects.all()[:3]
        courses_serializer = CourseSerializer(courses, many=True, context={'request': request})

        # Get the first 3 records from ModelB
        books = Book.objects.filter(collection__title_en="Essentials").order_by('order')[:10]
        books_serializer = BookCoverTagSerializer(books, many=True, context={'request': request})

        # Create a combined response
        response_data = {
            'courses': courses_serializer.data,
            'books': books_serializer.data
        }

        return Response(response_data)

# Generic view for creating CustomerMail
class CustomerMailCreateView(generics.CreateAPIView):
    queryset = CustomerMail.objects.all()
    serializer_class = CustomerMailSerializer

# Generic view for creating CustomerHappinessRecord
class CustomerHappinessRecordCreateView(generics.CreateAPIView):
    queryset = CustomerHappinessRecord.objects.all()
    serializer_class = CustomerHappinessRecordSerializer

    def get(self, request, *args, **kwargs):
        # Calculate the total customer rate and the number of customers
        total_customer_rate = CustomerHappinessRecord.objects.aggregate(Sum('customer_rate'))['customer_rate__sum']
        total_customers = CustomerHappinessRecord.objects.count()

        # Calculate the average customer rate
        if total_customers > 0:
            average_rate = total_customer_rate / total_customers
        else:
            average_rate = 0

        # Return the average rate as a response
        return Response({
            "average_rate": average_rate
        }, status=status.HTTP_200_OK)
    
class ContactUsCreateView(generics.CreateAPIView):
    queryset = ContactUs.objects.all()
    serializer_class = ContactUsSerializer

class CourseCollectionListView(generics.ListAPIView):
    permission_classes = [APIKeyPermission]
    queryset = CourseCollection.objects.order_by('order')
    serializer_class = CourseCollectionSerializer


class MarkVideoWatchedAPIView(APIView):
    """
    API to mark a video as watched by the user.
    Checks enrollment and previous video watch requirements,
    then updates user's watched videos and course progress.
    """
    permission_classes = [IsEnrolledInCoursePerm, HasWatchedPreviousVideoPerm]
    
    def post(self, request):
        # Get video_id from request data
        serializer = MarkVideoWatchedSerializer(data=request.data)
        
        if not serializer.is_valid():
            return Response({
                'status': False,
                'message': 'Invalid video ID',
                'errors': serializer.errors
            }, status=status.HTTP_400_BAD_REQUEST)
        
        video_id = serializer.validated_data['video_id']
        
        try:
            # Get the video object
            video = Video.objects.select_related('Course').get(id=video_id)
            user = request.user
            
            # Check if video is already watched (avoid duplicates)
            if user.user_watched_video.filter(id=video_id).exists():
                return Response({
                    'status': True,
                    'message': 'Video already marked as watched'
                }, status=status.HTTP_200_OK)
            
            # Add video to user's watched videos
            user.user_watched_video.add(video)
            
            # Calculate progress using the serializer method
            progress = serializer.get_course_progress(user, video.Course)
            
            # Check if course is 100% complete
            if progress >= 100:
                # Mark course as finished in UserCourse
                user_course = UserCourse.objects.get(user=user, course=video.Course)
                if not user_course.is_finished:
                    # implement PDU request functionality
                    user_course.is_finished = True
                    user_course.finished_date = timezone.now()
                    user_course.save()
            
            return Response({
                'status': True,
                'message': 'Video marked as watched successfully'
            }, status=status.HTTP_200_OK)
            
        except Video.DoesNotExist:
            return Response({
                'status': False,
                'message': 'Video not found'
            }, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            return Response({
                'status': False,
                'message': 'An error occurred while marking video as watched'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)





class CourseQuizAPIView(generics.RetrieveAPIView):
    """
    API to retrieve quiz questions for a course.
    Requires user to be enrolled in course and have finished the last video.
    """
    permission_classes = [IsEnrolledInCoursePerm, CanAccessCourseQuizPerm]
    queryset = Course.objects.all()
    serializer_class = CourseQuizSerializer
    lookup_field = 'pk'
    lookup_url_kwarg = 'course_id'


class QuizSubmissionAPIView(APIView):
    """
    API to submit quiz answers and calculate results.
    Handles all business logic: validation, scoring, and database updates.
    Course ID comes from request body, not URL.
    """
    permission_classes = [IsEnrolledInCoursePerm, CanAccessCourseQuizPerm]
    
    def post(self, request):
        """
        Submit quiz answers with course_id and answers from request body
        Expected format:
        {
            "course_id": 1,
            "answers": [
                {"question_id": 1, "answer": "A"},
                {"question_id": 2, "answer": "B"}
            ]
        }
        """
        # Basic validation with serializer
        serializer = QuizSubmissionSerializer(data=request.data)
        
        if not serializer.is_valid():
            return Response({
                'status': False,
                'message': 'Invalid data provided',
                'errors': serializer.errors
            }, status=status.HTTP_400_BAD_REQUEST)
        
        try:
            course_id = serializer.validated_data['course_id']
            answers = serializer.validated_data['answers']
            
            # Get course and questions 
            course = Course.objects.get(id=course_id)
            questions = Question.objects.filter(course=course)
            
            # Calculate score
            total_questions = questions.count()
            correct_answers = 0
            question_answers = {q.id: q.right_choice for q in questions}
            
            for answer_data in answers:
                question_id = int(answer_data['question_id'])
                user_answer = answer_data['answer']
                
                if question_answers.get(question_id) == user_answer:
                    correct_answers += 1
            
            # Calculate percentage and determine success
            score_percentage = (correct_answers / total_questions * 100) if total_questions > 0 else 0
            is_success = score_percentage > 50
            
            # Update UserCourse
            user_course, created = UserCourse.objects.get_or_create(
                user=request.user,
                course=course,
                defaults={'quiz_grades': 0, 'tries': 0}
            )
            
            user_course.tries += 1
            user_course.quiz_grades = int(score_percentage)
            
            if is_success:
                # check if user_coursse is not finshed implement PDU request functionality
                user_course.is_finished = True
                user_course.finished_date = timezone.now()
                
            user_course.save()
            
            return Response({
                'status': True,
                'message': 'Quiz submitted successfully',
                'data': {
                    'total_questions': total_questions,
                    'correct_answers': correct_answers,
                    'score_percentage': round(score_percentage, 2),
                    'result': 'success' if is_success else 'failed',
                    'quiz_grades': user_course.quiz_grades,
                    'tries': user_course.tries,
                    'is_finished': user_course.is_finished,
                    'finished_date': user_course.finished_date
                }
            }, status=status.HTTP_200_OK)
            
        except Exception as e:
            return Response({
                'status': False,
                'message': 'An error occurred while processing quiz submission',
                'error': str(e)
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class CourseEnrollmentAPIView(APIView):
    """
    API to enroll user in a course.
    """
    permission_classes = [IsAuthenticated]
    
    def post(self, request):
        serializer = CourseEnrollmentSerializer(data=request.data)
        if serializer.is_valid():
            course_id = serializer.validated_data['course_id']
            course = Course.objects.get(id=course_id)
            
            # Check if already enrolled (business logic in view)
            if UserCourse.objects.filter(user=request.user, course=course).exists():
                return Response({'error': 'Already enrolled'}, status=400)
            
            # Create enrollment
            UserCourse.objects.create(user=request.user, course=course)
            
            return Response({
                'status': True,
                'message': 'Successfully enrolled in course'
            }, status=status.HTTP_201_CREATED)
        
        return Response({
            'status': False,
            'errors': serializer.errors
        }, status=status.HTTP_400_BAD_REQUEST)