# serializers.py

from rest_framework import serializers
from .models import *
from django.db.models import Sum
from library.models import Tag, Language, TargetAudience
from .models import CourseCollection, UserCourse
from library.serializers import LanguageSerializer, TargetAudienceSerializer, TagSerializer
from library.utils import APIFileURLModelSerializer


class CourseSerializer(APIFileURLModelSerializer):
    video_count = serializers.SerializerMethodField()
    total_duration = serializers.SerializerMethodField()
    course_progress = serializers.SerializerMethodField()
    tag = TagSerializer()
    course_lang = LanguageSerializer(many=True)
    subtitle_lang = LanguageSerializer(many=True)
    target_audience = TargetAudienceSerializer(many=True)

    class Meta:
        model = Course
        fields = ['id', 'title_ar', 'title_en', 'about_ar', 'about_en', 'brief_about_en', 'brief_about_ar', 'video_count', 
                  'cover_photo', 'total_duration', 'views', 'tag', 'production_date', 'course_lang', 'subtitle_lang', 'target_audience','allowed_account_ids' , 'pdus', 'course_progress']

    def get_video_count(self, obj):
        return obj.videos.count()
    
    def get_total_duration(self, obj):
        # Aggregate the total duration from the related videos
        total_duration = obj.videos.aggregate(total_duration=Sum('duration'))['total_duration']
        
        if not total_duration:
            return "00:00"

        # Convert total duration (which is a timedelta) to total seconds
        total_seconds = total_duration.total_seconds()
        
        # Calculate hours and minutes
        hours = int(total_seconds // 3600)
        minutes = int((total_seconds % 3600) // 60)

        # Format hours and minutes as two-digit numbers
        return f"{hours:02}:{minutes:02}"
    
    def get_course_progress(self, obj):
        """Calculate course progress percentage"""
        request = self.context.get('request')
        if not request or not request.user.is_authenticated:
            return 0
            
        user = request.user
        
        # Get total videos in the course
        total_videos = obj.videos.count()
        
        # If no videos, return 0
        if total_videos == 0:
            return 0
        
        # Get watched videos for this user in this course
        watched_videos = user.user_watched_video.filter(Course=obj).count()
        
        # Calculate total items (videos + quiz if exists)
        # Use the has_quiz field to determine if course has a quiz
        has_actual_quiz = obj.has_quiz
        total_items = total_videos + (1 if has_actual_quiz else 0)
        
        # Check if quiz is completed (grade > 50)
        quiz_completed = 0
        if has_actual_quiz:
            user_course = UserCourse.objects.filter(user=user, course=obj).first()
            if user_course and user_course.quiz_grades and user_course.quiz_grades > 50:
                quiz_completed = 1
        
        # Calculate progress percentage including quiz completion
        completed_items = watched_videos + quiz_completed
        progress = (completed_items / total_items) * 100
        
        return round(progress)

class VideoSerializer(APIFileURLModelSerializer):
    video_lang = LanguageSerializer(many=True)

    class Meta:
        model = Video
        fields = '__all__'


class QuestionSerializer(serializers.ModelSerializer):
    class Meta:
        model = Question
        fields = [
            'id',
            'question',
            'timestamp',
            # 'order',      
            'choice_a',
            'choice_b',
            'choice_c',
            'choice_d',
            'right_choice',
        ]


class VideoQuestionSerializer(APIFileURLModelSerializer):
    video_lang = LanguageSerializer(many=True)
    questions = serializers.SerializerMethodField()  

    class Meta:
        model = Video
        fields = '__all__'

    def get_questions(self, obj):
        # Return all questions related to this video
        questions = obj.video_questions.all()  
        return QuestionSerializer(questions, many=True, context=self.context).data


class CourseWithVideosSerializer(APIFileURLModelSerializer):
    videos = serializers.SerializerMethodField()
    course_lang = LanguageSerializer(many=True)
    subtitle_lang = LanguageSerializer(many=True)
    total_duration = serializers.SerializerMethodField()
    target_audience = TargetAudienceSerializer(many=True)
    tag = TagSerializer()
    watched_videos = serializers.SerializerMethodField()
    first_video_questions = serializers.SerializerMethodField()
    course_progress = serializers.SerializerMethodField()

    class Meta:
        model = Course
        fields = [
            'id',
            'title_ar',
            'title_en',
            'about_ar',
            'about_en',
            'brief_about_en',
            'brief_about_ar',
            'videos',
            'cover_photo',
            'views',
            'tag',
            'production_date',
            'course_lang',
            'subtitle_lang',
            'total_duration',
            'target_audience',
            'allowed_account_ids',
            'pdus',
            "has_quiz",
            'watched_videos',
            'first_video_questions',
            'course_progress'
        ]

    def get_videos(self, obj):
        # Retrieve videos related to this course, ordered by the 'order' field
        ordered_videos = obj.videos.order_by('order')
        return VideoSerializer(ordered_videos, many=True, context=self.context).data
    
    def get_total_duration(self, obj):
        # Aggregate the total duration from the related videos
        total_duration = obj.videos.aggregate(total_duration=Sum('duration'))['total_duration']
        
        if not total_duration:
            return "00:00"

        # Convert total duration (which is a timedelta) to total seconds
        total_seconds = total_duration.total_seconds()
        
        # Calculate hours and minutes
        hours = int(total_seconds // 3600)
        minutes = int((total_seconds % 3600) // 60)

        # Format hours and minutes as two-digit numbers
        return f"{hours:02}:{minutes:02}"

    def get_watched_videos(self, obj):
        """Return list of watched video objects (id + title_en) for this course by current user"""
        request = self.context.get("request")
        if not request or not request.user.is_authenticated:
            return []

        user = request.user
        watched_videos = user.user_watched_video.filter(Course=obj).values("id", "title_en")
        return list(watched_videos)
    
    def get_first_video_questions(self, obj):
        """Return questions for the first video in the course"""
        # Get the first video ordered by 'order' field
        first_video = obj.videos.order_by('order').first()
        
        if not first_video:
            return []
        
        # Get questions related to the first video
        questions = first_video.video_questions.all()
        return QuestionSerializer(questions, many=True, context=self.context).data
    
    def get_course_progress(self, obj):
        """Calculate course progress percentage"""
        request = self.context.get('request')
        if not request or not request.user.is_authenticated:
            return 0
            
        user = request.user
        
        # Get total videos in the course
        total_videos = obj.videos.count()
        
        # If no videos, return 0
        if total_videos == 0:
            return 0
        
        # Get watched videos for this user in this course
        watched_videos = user.user_watched_video.filter(Course=obj).count()
        
        # Calculate total items (videos + quiz if exists)
        # Use the has_quiz field to determine if course has a quiz
        has_actual_quiz = obj.has_quiz
        total_items = total_videos + (1 if has_actual_quiz else 0)
        
        # Check if quiz is completed (grade > 50)
        quiz_completed = 0
        if has_actual_quiz:
            user_course = UserCourse.objects.filter(user=user, course=obj).first()
            if user_course and user_course.quiz_grades and user_course.quiz_grades > 50:
                quiz_completed = 1
        
        # Calculate progress percentage including quiz completion
        completed_items = watched_videos + quiz_completed
        progress = (completed_items / total_items) * 100
        
        return round(progress)



class TempSerializer(APIFileURLModelSerializer):
    class Meta:
        model = Temporary
        fields = '__all__'



class CustomerMailSerializer(APIFileURLModelSerializer):
    class Meta:
        model = CustomerMail
        fields = ['id', 'customer_mail'] 


class CustomerHappinessRecordSerializer(APIFileURLModelSerializer):
    class Meta:
        model = CustomerHappinessRecord
        fields = ['id', 'customer_rate']

class ContactUsSerializer(APIFileURLModelSerializer):
    class Meta:
        model = ContactUs
        fields = '__all__'


class CourseCollectionSerializer(APIFileURLModelSerializer):
    class Meta:
        model = CourseCollection
        fields = ['id', 'title_en', 'title_ar']


class MarkVideoWatchedSerializer(serializers.Serializer):
    video_id = serializers.IntegerField()
    
    def validate_video_id(self, value):
        """Validate that the video exists"""
        try:
            video = Video.objects.get(id=value)
            return value
        except Video.DoesNotExist:
            raise serializers.ValidationError("Video not found")
    
    def get_course_progress(self, user, course):
        """Calculate course progress percentage - for internal checking only"""
        # Get total videos in the course
        total_videos = course.videos.count()
        
        # If no videos, return 0
        if total_videos == 0:
            return 0
        
        # Get watched videos for this user in this course
        watched_videos = user.user_watched_video.filter(Course=course).count()
        
        # Calculate total items (videos + quiz if exists)
        # Use the has_quiz field to determine if course has a quiz
        has_actual_quiz = course.has_quiz
        total_items = total_videos + (1 if has_actual_quiz else 0)
        
        # Check if quiz is completed (grade > 50)
        quiz_completed = 0
        if has_actual_quiz:
            user_course = UserCourse.objects.filter(user=user, course=course).first()
            if user_course and user_course.quiz_grades and user_course.quiz_grades > 50:
                quiz_completed = 1
        
        # Calculate progress percentage including quiz completion
        completed_items = watched_videos + quiz_completed
        progress = (completed_items / total_items) * 100
        
        return round(progress)


class QuizQuestionSerializer(serializers.ModelSerializer):
    """Serializer for quiz questions with choices A-D (excludes right_choice)"""
    class Meta:
        model = Question
        fields = [
            'id',
            'question',
            'choice_a',
            'choice_b', 
            'choice_c',
            'choice_d'
        ]


class CourseQuizSerializer(serializers.ModelSerializer):
    """Serializer for course quiz containing all questions"""
    questions = serializers.SerializerMethodField()
    total_questions = serializers.SerializerMethodField()
    
    class Meta:
        model = Course
        fields = [
            'id',
            'title_ar',
            'title_en', 
            'has_quiz',
            'questions',
            'total_questions'
        ]
    
    def get_questions(self, obj):
        """Get all quiz questions for this course"""
        # Only return questions if course has_quiz flag is True
        if not obj.has_quiz:
            return []
        questions = obj.course_questions.all()
        return QuizQuestionSerializer(questions, many=True, context=self.context).data
    
    def get_total_questions(self, obj):
        """Get total number of questions in the quiz"""
        # Only count questions if course has_quiz flag is True
        if not obj.has_quiz:
            return 0
        return obj.course_questions.count()


class QuizSubmissionSerializer(serializers.Serializer):
    """Simple serializer for quiz submission validation"""
    course_id = serializers.IntegerField()
    answers = serializers.ListField(
        child=serializers.DictField(child=serializers.CharField()),
        help_text="List of answers in format: [{'question_id': '1', 'answer': 'A'}, ...]"
    )
    
    def validate_course_id(self, value):
        """Validate that the course exists"""
        try:
            Course.objects.get(id=value)
            return value
        except Course.DoesNotExist:
            raise serializers.ValidationError("Course not found")
    
    def validate(self, data):
        """Validate that all question IDs belong to the specified course"""
        course_id = data.get('course_id')
        answers = data.get('answers', [])
        
        if course_id and answers:
            # Get all question IDs from the course
            course_question_ids = set(
                Question.objects.filter(course_id=course_id).values_list('id', flat=True)
            )
            
            # Check each question ID in the answers
            for answer in answers:
                question_id = answer.get('question_id')
                if question_id:
                    try:
                        question_id = int(question_id)
                        if question_id not in course_question_ids:
                            raise serializers.ValidationError(
                                f"Question ID {question_id} does not belong to course {course_id}"
                            )
                    except (ValueError, TypeError):
                        raise serializers.ValidationError(
                            f"Invalid question_id format: {question_id}"
                        )
        
        return data


class CourseEnrollmentSerializer(serializers.Serializer):
    course_id = serializers.IntegerField()
    
    def validate_course_id(self, value):
        """Validate that the course exists"""
        try:
            Course.objects.get(id=value)
            return value
        except Course.DoesNotExist:
            raise serializers.ValidationError("Course not found")