from rest_framework.permissions import BasePermission
from django.conf import settings
from rest_framework.exceptions import APIException, PermissionDenied
from rest_framework import status
from .models import Video, UserCourse , Course



class InvalidAPIKey(APIException):
    status_code = status.HTTP_401_UNAUTHORIZED
    default_detail = {
        'status': False,
        'message': 'Unauthorized API.'
    }
    default_code = 'unauthorized'


class APIKeyPermission(BasePermission):
    def has_permission(self, request, view):
        expected_key = getattr(settings, 'API_KEY', None)
        api_key = request.headers.get('X-API-KEY')

        if not expected_key:
            raise APIException('Server misconfigured: missing API key.', code='server_error')

        if api_key != expected_key:
            raise InvalidAPIKey()

        return True


class IsEnrolledInCoursePerm(BasePermission):
    """
    Permission to check if user is enrolled in the course.
    Supports both video_id and course_id from URL kwargs and request data.
    """
    def has_permission(self, request, view):
        # Check if user is authenticated
        if not request.user or not request.user.is_authenticated:
            raise PermissionDenied("You must be authenticated to access this resource")
        
        # Try to get course_id first, then video_id
        course_id = view.kwargs.get('course_id')
        video_id = view.kwargs.get('video_id') or view.kwargs.get('pk')
        
        # If no course_id, try to get from request data
        if not course_id and hasattr(request, 'data'):
            course_id = request.data.get('course_id')
        
        # If no video_id, try to get video_id from request data
        if not course_id and not video_id and hasattr(request, 'data'):
            video_id = request.data.get('video_id')
        
        course = None
        
        try:
            if course_id:
                # Direct course access
                course = Course.objects.get(id=course_id)
            elif video_id:
                # Get course through video
                video = Video.objects.select_related('Course').get(id=video_id)
                course = video.Course
            else:
                raise PermissionDenied("Course ID or Video ID is required")
            
            # Check if user is enrolled in the course
            is_enrolled = UserCourse.objects.filter(
                user=request.user,
                course=course
            ).exists()
            
            if not is_enrolled:
                raise PermissionDenied("You are not enrolled in this course")
            
            return True
            
        except Course.DoesNotExist:
            raise PermissionDenied("Course not found")
        except Video.DoesNotExist:
            raise PermissionDenied("Video not found")


class HasWatchedPreviousVideoPerm(BasePermission):
    """
    Permission to check if user has watched the previous video in the course.
    If it's the first video (lowest order), access is allowed.
    Supports video_id from both URL kwargs and request data.
    """
    def has_permission(self, request, view):
        # Check if user is authenticated
        if not request.user or not request.user.is_authenticated:
            raise PermissionDenied("You must be authenticated to access this video")
        
        # Get video_id from URL kwargs first, then from request data
        video_id = view.kwargs.get('video_id') or view.kwargs.get('pk')
        if not video_id and hasattr(request, 'data'):
            video_id = request.data.get('video_id')
        
        if not video_id:
            raise PermissionDenied("Video ID is required")
        
        try:
            current_video = Video.objects.select_related('Course').get(id=video_id)
            
            # Find the previous video in the same course (by order)
            previous_video = Video.objects.filter(
                Course=current_video.Course,
                order__lt=current_video.order
            ).order_by('-order').first()
            
            # If no previous video exists (this is the first video), allow access
            if not previous_video:
                return True
            
            # Check if user has watched the previous video
            has_watched = request.user.user_watched_video.filter(id=previous_video.id).exists()
            
            if not has_watched:
                raise PermissionDenied("You must watch the previous video before accessing this one")
            
            return True
            
        except Video.DoesNotExist:
            raise PermissionDenied("Video not found")


class CanAccessCourseQuizPerm(BasePermission):
    """
    Permission to check if user can access course quiz.
    Checks:
    1. Course has quiz enabled
    2. User has finished the last video in the course
    Supports course_id from both URL kwargs and request data.
    """
    def has_permission(self, request, view):
        # Check if user is authenticated
        if not request.user or not request.user.is_authenticated:
            raise PermissionDenied("You must be authenticated to access the quiz")
        
        # Get course_id from URL kwargs first, then from request data
        course_id = view.kwargs.get('course_id') or view.kwargs.get('pk')
        if not course_id and hasattr(request, 'data'):
            course_id = request.data.get('course_id')
        
        if not course_id:
            raise PermissionDenied("Course ID is required")
        
        try:
            course = Course.objects.get(id=course_id)
            
            # Check if course has quiz enabled
            if not course.has_quiz:
                raise PermissionDenied("This course does not have a quiz")
            
            # Get the last video in the course (highest order)
            last_video = course.videos.order_by('-order').first()
            
            if not last_video:
                raise PermissionDenied("No videos found in this course")
            
            # Check if user has watched the last video
            has_watched_last_video = request.user.user_watched_video.filter(
                id=last_video.id
            ).exists()
            
            if not has_watched_last_video:
                raise PermissionDenied("You must finish watching all videos before accessing the quiz")
            
            return True
            
        except Course.DoesNotExist:
            raise PermissionDenied("Course not found")

