from django_filters import rest_framework as filters
from .models import Course
from django.db.models import Q, Value, BooleanField
from django.db.models.functions import Lower

class CourseFilter(filters.FilterSet):
    search = filters.CharFilter(method='filter_by_all')
    course_lang = filters.CharFilter(method='filter_by_course_lang')
    price_gt = filters.NumberFilter(field_name="price", lookup_expr='gt')
    price_e = filters.NumberFilter(field_name="price")
    target_audience = filters.CharFilter(method='filter_by_trg_audience')
    collection = filters.NumberFilter(method='filter_by_collection')
    registered = filters.BooleanFilter(method='filter_by_registered')
    sort_by = filters.OrderingFilter(
        fields=(
            ('views', 'views'),
            ('order', 'order'),
            ('production_date', 'production_date'),
            ('title_en', 'title_en'),
            ('title_ar', 'title_ar')
        )
    )

    class Meta:
        model = Course
        fields = ['price','course_lang']

    def filter_by_all(self, queryset, name, value):
        # Split the input value into separate words
        search_terms = value.split()

        # Step 1: Filter for exact matches of the whole value in any of the fields
        whole_value_query = (
            Q(title_ar__icontains=value) |
            Q(title_en__icontains=value) |
            Q(about_ar__icontains=value) |
            Q(about_en__icontains=value) |
            Q(videos__title_ar__icontains=value) |
            Q(videos__title_en__icontains=value)
        )

        # Step 2: Build a Q object for individual word matches
        term_query = Q()
        for term in search_terms:
            term_query |= (
                Q(title_ar__icontains=term) |
                Q(title_en__icontains=term) |
                Q(about_ar__icontains=term) |
                Q(about_en__icontains=term) |
                Q(videos__title_ar__icontains=term) |
                Q(videos__title_en__icontains=term)
            )

        # Step 3: Combine whole-value query with the individual terms query
        combined_query = whole_value_query | term_query

        # Step 4: Annotate the queryset with a boolean flag to prioritize exact matches
        queryset = queryset.annotate(
            is_exact_match=Value(True, output_field=BooleanField())
        ).filter(combined_query).distinct()

        # Step 5: Order by the annotation so that exact matches come first
        return queryset.order_by('-is_exact_match')
        
    def filter_by_course_lang(self, queryset, name, value):
        return queryset.filter(
            course_lang__lang__icontains=value
        )
    
    def filter_queryset(self, queryset):

        queryset = super().filter_queryset(queryset)
        
        # Get the ordering parameter from the request
        ordering = self.data.get('sort_by')
        if ordering:
            if 'title_en' in ordering:
                # Annotate with case-insensitive field and then order by it
                queryset = queryset.annotate(lower_title_en=Lower('title_en'))
                if ordering.startswith('-'):
                    queryset = queryset.order_by('-lower_title_en')
                else:
                    queryset = queryset.order_by('lower_title_en')
            elif 'title_ar' in ordering:
                # Annotate with case-insensitive field and then order by it
                queryset = queryset.annotate(lower_title_ar=Lower('title_ar'))
                if ordering.startswith('-'):
                    queryset = queryset.order_by('-lower_title_ar')
                else:
                    queryset = queryset.order_by('lower_title_ar')
            else:
                # Fall back to default ordering
                queryset = queryset.order_by(*ordering.split(','))

        return queryset

    def filter_by_trg_audience(self, queryset, name, value):
        return queryset.filter(
            target_audience__title_en__exact=value
        )

    def filter_by_collection(self, queryset, name, value):
        return queryset.filter(collection__id=value)

    def filter_by_registered(self, queryset, name, value):
        if value:
            return queryset.filter(registered_users=self.request.user)
        return queryset

